/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.core;

import java.net.URI;
import java.time.Duration;
import java.util.Collections;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.nimbusds.oauth2.sdk.ResponseMode;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.State;
import com.nimbusds.openid.connect.sdk.Display;
import com.nimbusds.openid.connect.sdk.Nonce;
import com.nimbusds.openid.connect.sdk.Prompt;
import com.nimbusds.openid.connect.sdk.claims.ACR;

import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * An OAuth 2.0 authorization request.
 * <p> This class is mutable and not thread-safe.</p> 
 */
public class OAuthAuthorizationRequest {
    
    /** The client identifier. Required.*/
    @Nonnull private final ClientID clientID;
    
    /** The state. */
    @Nullable private State state;
    
    /** The redirect URI to which the response will be sent. */
    @Nullable private URI redirectURI;
    
    /** The authorization processing flow to use.*/
    @Nullable private ResponseType responseType;
    
    /** The request endpoint. */
    @Nullable private URI endpointURI;
    
    /** The nonce. */
    @Nullable private Nonce nonce;
    
    /** The requested scopes.*/
    @Nonnull private final Scope scope;
    
    /**
     * ASCII string value that specifies how the Authorization Server 
     * displays the authentication and consent user interface pages to the End-User.
     */
    @Nullable private Display display;
    
    /**
     * Space delimited, case sensitive list of ASCII string values that 
     * specifies whether the Authorization Server prompts the End-User 
     * for reauthentication and consent.
     */
    @Nullable private Prompt prompt;
    
    /** 
     * Specifies the allowable elapsed time in seconds since the last time 
     * the End-User was actively authenticated by the OP.
     */
    @Nullable private Duration maxAge;
    
    /** 
     * List of requested authentication context class reference values. 
     * Values appear in order of preference. Optional. 
     */
    @Nonnull @NotLive private List<ACR> acrs;
    
    /** The response mode. Optional. */
    @Nullable private ResponseMode responseMode;
    
    //TODO there are a few others.
    
    /**
     * 
     * Constructor.
     *
     * @param id the clientID. Never {@literal null}.
     */
    public OAuthAuthorizationRequest(@Nonnull final ClientID id) {        
        clientID = Constraint.isNotNull(id, "ClientID can not be null");
        acrs = Collections.emptyList();
        scope = new Scope();
        
    }

    /**
     * Get the client ID. 
     * 
     * @return the clientID.
     */
    @Nonnull public ClientID getClientID() {
        return clientID;
    }

    /**
     * Get the state.
     * 
     * @return the state.
     */
    @Nullable public State getState() {
        return state;
    }

    /**
     * Set the state. Optional. 
     * 
     * @param theState The state to set.
     */
    public void setState(@Nullable final State theState) {
        state = theState;
    }

    /**
     * Get the redirect_uri.
     * 
     * @return the redirectURI.
     */
    @Nullable public URI getRedirectURI() {
        return redirectURI;
    }

    /**
     * Set the redirect_uri. 
     * 
     * @param uri The redirect_uri to set.
     */
    public void setRedirectURI(@Nonnull final URI uri) {
        redirectURI = Constraint.isNotNull(uri,"RedirectURI can not be null");
    }

    /**
     * Get the response type.
     * 
     * @return the responseType.
     */
    @Nullable public ResponseType getResponseType() {
        return responseType;
    }

    /**
     * Set the response type.
     * 
     * @param type The responseType to set.
     */
    public void setResponseType(@Nonnull final ResponseType type) {
        responseType = Constraint.isNotNull(type,"ResponseType can not be null");
    }

    /**
     * Get the endpoint_uri.
     * 
     * @return the endpoint_uri.
     */
    @Nullable public URI getEndpointURI() {
        return endpointURI;
    }

    /**
     * Set the  endpoint_uri.
     * 
     * @param uri The endpointURI to set.
     */
    public void setEndpointURI(@Nonnull final URI uri) {
        endpointURI =  Constraint.isNotNull(uri,"EndpointURI can not be null");
    }

    /**
     * Get the nonce. 
     * 
     * @return the nonce.
     */
    @Nullable public Nonce getNonce() {
        return nonce;
    }

    /**
     * Set the nonce. 
     * 
     * @param theNonce The nonce to set.
     */
    public void setNonce(@Nullable final Nonce theNonce) {
        nonce = theNonce;
    }

    /**
     * Get the scope. 
     * 
     * @return the scope.
     */
    @Nonnull public Scope getScope() {
        return scope;
    }

    /**
     * Get the display.
     * 
     * @return the display.
     */
    @Nullable public Display getDisplay() {
        return display;
    }

    /**
     * Set the display. 
     * 
     * @param theDisplay The display to set.
     */
    public void setDisplay(@Nullable final Display theDisplay) {
        display = theDisplay;
    }

    /**
     * Get the prompt. 
     * 
     * @return the prompt.
     */
    @Nullable public Prompt getPrompt() {
        return prompt;
    }

    /**
     * Set the prompt. 
     * 
     * @param thePrompt The prompt to set.
     */
    public void setPrompt(@Nullable final Prompt thePrompt) {
        prompt = thePrompt;
    }

    /**
     * Get the max age.
     * 
     * @return the maxAge.
     */
    @Nullable public Duration getMaxAge() {
        return maxAge;
    }

    /**
     * Set the max age.
     * 
     * @param max The maxAge to set.
     */
    public void setMaxAge(@Nullable final Duration max) {
        maxAge = max;
    }

    /**
     * Get the ACRs.
     * 
     * @return the acrs.
     */
    @Nonnull @NotLive @Unmodifiable public List<ACR> getAcrs() {
        return Collections.unmodifiableList(acrs);
    }

    /**
     * Set the ACRs.
     * 
     * @param theAcrs The acrs to set.
     */
    public void setAcrs(@Nullable final List<ACR> theAcrs) {
        if (theAcrs != null) {
            acrs = Collections.unmodifiableList(theAcrs);
        }
    }

    /**
     * Get the response mode.
     * 
     * @return the responseMode.
     */
    @Nullable public ResponseMode getResponseMode() {
        return responseMode;
    }

    /**
     * Set the response mode.
     * 
     * @param mode The responseMode to set.
     */
    public void setResponseMode(@Nullable final ResponseMode mode) {
        responseMode = mode;
    }


}
