/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config.navigate;

import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;

import net.shibboleth.idp.profile.config.ProfileConfiguration;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.idp.profile.context.navigate.AbstractRelyingPartyLookupFunction;
import net.shibboleth.oidc.profile.oauth2.config.AbstractOAuth2ClientAuthenticableProfileConfiguration;

/**
 * A function that obtains
 * {@link AbstractOAuth2ClientAuthenticableProfileConfiguration#getTokenEndpointAuthMethods(ProfileRequestContext)}
 * if such a profile is available from a {@link RelyingPartyContext} obtained via a lookup function,
 * by default a child of the {@link ProfileRequestContext}. That result is then transformed into a list
 * of {@link ClientAuthenticationMethod}s.
 * 
 * <p>If a specific setting is unavailable, a null value is returned.</p>
 */
public class TokenEndpointAuthMethodLookupFunction
        extends AbstractRelyingPartyLookupFunction<Set<ClientAuthenticationMethod>> {

    /** {@inheritDoc} */
    @Nullable public Set<ClientAuthenticationMethod> apply(@Nullable final ProfileRequestContext input) {
        final RelyingPartyContext rpc = getRelyingPartyContextLookupStrategy().apply(input);
        if (rpc != null) {
            final ProfileConfiguration pc = rpc.getProfileConfig();
            if (pc instanceof AbstractOAuth2ClientAuthenticableProfileConfiguration) {
                final Set<String> methods =
                        ((AbstractOAuth2ClientAuthenticableProfileConfiguration)pc).getTokenEndpointAuthMethods(input);
                if (methods != null) {
                    return methods
                            .stream()
                            .map(ClientAuthenticationMethod::new)
                            .collect(Collectors.toUnmodifiableSet());
                }
            }
        }
        
        return null;
    }

}