/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config.navigate;

import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.nimbusds.oauth2.sdk.GrantType;

import net.shibboleth.idp.profile.config.ProfileConfiguration;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.idp.profile.context.navigate.AbstractRelyingPartyLookupFunction;
import net.shibboleth.oidc.profile.config.OIDCAuthorizationConfiguration;
import net.shibboleth.oidc.profile.oauth2.config.OAuth2TokenConfiguration;

/**
 * A function that obtains
 * {@link OAuth2TokenConfiguration#getGrantTypes(ProfileRequestContext)}
 * if such a profile is available from a {@link RelyingPartyContext} obtained via a lookup function,
 * by default a child of the {@link ProfileRequestContext}. That result is then transformed into a list
 * of {@link GrantType}s.
 * 
 * <p>If a specific setting is unavailable, a null value is returned.</p>
 */
public class GrantTypesLookupFunction extends AbstractRelyingPartyLookupFunction<Set<GrantType>> {

    /** {@inheritDoc} */
    @Nullable public Set<GrantType> apply(@Nullable final ProfileRequestContext input) {
        final RelyingPartyContext rpc = getRelyingPartyContextLookupStrategy().apply(input);
        if (rpc != null) {
            final ProfileConfiguration pc = rpc.getProfileConfig();
            if (pc instanceof OAuth2TokenConfiguration) {
                final Set<String> types = ((OAuth2TokenConfiguration)pc).getGrantTypes(input);
                if (types != null) {
                    return types.stream()
                            .map(GrantType::new)
                            .collect(Collectors.toUnmodifiableSet());
                }
            } else if (pc instanceof OIDCAuthorizationConfiguration) {
                // For compatibility just return the lone supported type.
                return Set.of(GrantType.AUTHORIZATION_CODE, GrantType.REFRESH_TOKEN);
            }
        }
        
        return null;
    }

}