/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.profile.config.AbstractConditionalProfileConfiguration;
import net.shibboleth.idp.profile.config.OverriddenIssuerProfileConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;

/**
 * Profile configuration for the OpenID Connect Provider Configuration.
 */
public class OIDCProviderInformationConfiguration extends AbstractConditionalProfileConfiguration
        implements OIDCProfileConfiguration, OverriddenIssuerProfileConfiguration {

    /** OIDC base protocol URI. Section 4 is relevant. */
    public static final String PROTOCOL_URI = "http://openid.net/specs/openid-connect-discovery-1_0.html";

    /** ID for this profile configuration. */
    public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oidc/configuration";

    /** Lookup function to override issuer value. */
    @Nonnull private Function<ProfileRequestContext,String> issuerLookupStrategy;
    
    /**
     * Constructor.
     */
    public OIDCProviderInformationConfiguration() {
        this(PROFILE_ID);
    }
    
    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public OIDCProviderInformationConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
        issuerLookupStrategy = FunctionSupport.constant(null);
    }
    
    /** {@inheritDoc} */
    @Nullable @NotEmpty public String getIssuer(@Nullable final ProfileRequestContext profileRequestContext) {
        return issuerLookupStrategy.apply(profileRequestContext);
    }
    
    /**
     * Set overridden issuer value.
     * 
     * @param issuer issuer value
     */
    public void setIssuer(@Nullable @NotEmpty final String issuer) {
        issuerLookupStrategy = FunctionSupport.constant(issuer);
    }
    
    /**
     * Sets lookup strategy for overridden issuer value.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        issuerLookupStrategy = Constraint.isNotNull(strategy, "Issuer lookup strategy cannot be null");
    }

}