/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.base.Predicates;

import net.shibboleth.idp.authn.config.AuthenticationProfileConfiguration;
import net.shibboleth.idp.profile.config.OverriddenIssuerProfileConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Profile configuration for the OpenID Connect authorization endpoint.
 * 
 * <p>It is also usable as a Token endpoint configuration if no non-OIDC use cases are needed.</p>
 */
public class OIDCAuthorizationConfiguration extends AbstractOIDCSSOConfiguration
        implements OIDCProfileConfiguration, AuthenticationProfileConfiguration, OverriddenIssuerProfileConfiguration {

    /** ID for this profile configuration. */
    @Nonnull @NotEmpty public static final String PROFILE_ID = "http://shibboleth.net/ns/profiles/oidc/sso/browser";
    
    /** Whether all acr claim requests should be treated as Essential. */
    @Nonnull private Predicate<ProfileRequestContext> acrRequestAlwaysEssentialPredicate;

    /** Whether to encode consent in authorization code and access/refresh tokens. */
    @Nonnull private Predicate<ProfileRequestContext> encodeConsentInTokensPredicate;

    /** Lookup function to supply lifetime of authz code. */
    @Nonnull private Function<ProfileRequestContext,Duration> authorizeCodeLifetimeLookupStrategy;
    
    /** Lookup function to supply attribute IDs to embed in authorization code or access token. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> encodedAttributesLookupStrategy;
    
    /** Whether to encode authentication request parameters inside a JWT request object .*/
    @Nonnull private Predicate<ProfileRequestContext> useRequestObjectPredicate;

    /** Lookup function to supply attribute IDs to omit from UserInfo token. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> deniedUserInfoAttributesLookupStrategy;
    
    /** Whether to include iss parameter in the authentication response. */
    @Nonnull private Predicate<ProfileRequestContext> includeIssuerInResponsePredicate;

    /** Enumeration of the HTTP methods used in OIDC authentication requests.*/
    public enum OIDCHttpRequestMethod {    
        /**
         * HTTP GET.
         */
        GET,                
        /**
         * HTTP POST.
         */
        POST
    }
    
    /** 
     * Which HTTP method should be used to issue OIDC authentication requests. 
     * Supported values are POST and GET. The default is GET. 
     */
    @Nonnull private Function<ProfileRequestContext,String> httpRequestMethodLookupStrategy;

    /** Lookup function to supply strategy bi-function for manipulating authorization code claims set. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        authorizationCodeClaimsSetManipulationStrategyLookupStrategy;

    /**
     * Constructor.
     */
    public OIDCAuthorizationConfiguration() {
        this(PROFILE_ID);
    }

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public OIDCAuthorizationConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);

        acrRequestAlwaysEssentialPredicate = Predicates.alwaysFalse();
        encodeConsentInTokensPredicate = Predicates.alwaysFalse();
        useRequestObjectPredicate = Predicates.alwaysFalse();

        authorizeCodeLifetimeLookupStrategy = FunctionSupport.constant(Duration.ofMinutes(5));
        
        encodedAttributesLookupStrategy = FunctionSupport.constant(null);
        deniedUserInfoAttributesLookupStrategy = FunctionSupport.constant(null);
        httpRequestMethodLookupStrategy = FunctionSupport.constant(OIDCHttpRequestMethod.GET.toString());

        includeIssuerInResponsePredicate = Predicates.alwaysFalse();
        
        authorizationCodeClaimsSetManipulationStrategyLookupStrategy = FunctionSupport.constant(null);
    }

    /**
     * Get whether all acr claim requests should be treated as Essential.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether all acr claim requests should be treated as Essential
     */
    public boolean isAcrRequestAlwaysEssential(@Nullable final ProfileRequestContext profileRequestContext) {
        return acrRequestAlwaysEssentialPredicate.test(profileRequestContext);
    }

    /**
     * Set whether all acr claim requests should be treated as Essential.
     * 
     * @param flag flag to set
     */
    public void setAcrRequestAlwaysEssential(final boolean flag) {
        acrRequestAlwaysEssentialPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }
    
    /**
     * Set condition for whether all acr claim requests should be treated as Essential.
     * 
     * @param condition condition to set
     */
    public void setAcrRequestAlwaysEssentialPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        acrRequestAlwaysEssentialPredicate = Constraint.isNotNull(condition, "Condition cannot be null");
    }
    
    /**
     * Should authentication request parameters should be passed in a single, self contained, JWT?
     * 
     * @param profileRequestContext the profile request context
     * 
     * @return whether authentication request parameters should be passed in a single, self contained, JWT
     */
    public boolean isUseRequestObject(@Nullable final ProfileRequestContext profileRequestContext) {
        return useRequestObjectPredicate.test(profileRequestContext);
    }
    
    /**
     * Set whether the authentication request parameters should be passed in a single, self contained, JWT. 
     *
     * @param flag flag to set
     */
    public void setUseRequestObject(final boolean flag) {
        useRequestObjectPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition for whether the authentication request parameters should be passed in a single, 
     * self contained, JWT. 
     * 
     * @param condition condition to set
     */
    public void setUseRequestObjectPredicate(
            @Nonnull final Predicate<ProfileRequestContext> condition) {
        useRequestObjectPredicate = Constraint.isNotNull(condition, "Use request object condition cannot be null");
    }

    /**
     * Get whether to encode consent in authorization code and access/refresh tokens.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether to encode consent in authorization code and access/refresh tokens
     */
    public boolean isEncodeConsentInTokens(@Nullable final ProfileRequestContext profileRequestContext) {
        return encodeConsentInTokensPredicate.test(profileRequestContext);
    }

    /**
     * Set whether to encode consent in authorization code and access/refresh tokens.
     * 
     * @param flag flag to set
     */
    public void setEncodeConsentInTokens(final boolean flag) {
        encodeConsentInTokensPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition for whether to encode consent in authorization code and access/refresh tokens.
     * 
     * @param condition condition to set
     */
    public void setEncodeConsentInTokensPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        encodeConsentInTokensPredicate = Constraint.isNotNull(condition, "Condition cannot be null");
    }
    
    /**
     * Get authz code lifetime.
     * 
     * <p>Defaults to 5 minutes.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return authz code lifetime
     */
    @Positive @Nonnull
    public Duration getAuthorizeCodeLifetime(@Nullable final ProfileRequestContext profileRequestContext) {
        final Duration lifetime = authorizeCodeLifetimeLookupStrategy.apply(profileRequestContext);
        
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Authorization code lifetime must be greater than 0");
        return lifetime;
    }

    /**
     * Set the lifetime of authz code.
     * 
     * @param lifetime lifetime of authz code
     */
    public void setAuthorizeCodeLifetime(@Positive @Nonnull final Duration lifetime) {
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Authorization code lifetime must be greater than 0");
        
        authorizeCodeLifetimeLookupStrategy = FunctionSupport.constant(lifetime);
    }
    
    /**
     * Set a lookup strategy for the authz code lifetime.
     *
     * @param strategy lookup strategy
     */
    public void setAuthorizeCodeLifetimeLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        authorizeCodeLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }
    
    /**
     * Set a lookup strategy to determine the HTTP request method for an authentication request.
     * 
     * @param strategy the strategy to set.
     */
    public void setHttpRequestMethodLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, String> strategy) {
        httpRequestMethodLookupStrategy = 
                Constraint.isNotNull(strategy, "HTTP request method strategy can not be null");
    }
    
    /**
     * Set the HTTP request method for an authentication request.
     * 
     * @param method the HTTP method to set, either POST or GET.
     */
    public void setHttpRequestMethod(@Nonnull @NotEmpty final OIDCHttpRequestMethod method){
        httpRequestMethodLookupStrategy = FunctionSupport.constant(method != null ? method.toString() : null);
    }
    
    /**
     * Get the HTTP request method for an authentication request.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the HTTP request method
     */
    public OIDCHttpRequestMethod getHttpRequestMethod(@Nullable final ProfileRequestContext profileRequestContext) {
        final String method = httpRequestMethodLookupStrategy.apply(profileRequestContext);
        if (method != null) {
            try {
                return OIDCHttpRequestMethod.valueOf(method);
            } catch (final IllegalArgumentException e) {
                throw new ConstraintViolationException("Unexpected HTTP method value: '" + method + "': "
                        + e.getMessage());
            }
        }
        return null;
    }

    /**
     * Get the set of attribute IDs which should be encoded in encrypted form into the authorization code
     * and/or access/refresh tokens to enable recovery on the back-channel.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the attribute IDs to encode
     */
    @Nonnull @NonnullElements @NotLive public Set<String> getEncodedAttributes(
            @Nullable final ProfileRequestContext profileRequestContext) {
        
        final Set<String> attributes = encodedAttributesLookupStrategy.apply(profileRequestContext);
        if (attributes != null) {
            return Set.copyOf(attributes);
        }
        return Collections.emptySet();
    }

    /**
     * Set the set of attribute IDs which should be encoded in encrypted form into the authorization code
     * and/or access/refresh tokens to enable recovery on the back-channel.
     * 
     * @param attributes the attribute IDs to encode
     */
    public void setEncodedAttributes(@Nullable @NonnullElements final Collection<String> attributes) {

        if (attributes == null || attributes.isEmpty()) {
            encodedAttributesLookupStrategy = FunctionSupport.constant(null);
        } else {
            encodedAttributesLookupStrategy = FunctionSupport.constant(
                    Set.copyOf(StringSupport.normalizeStringCollection(attributes)));
        }
    }

    /**
     * Set a lookup strategy for the attribute IDs which should be encoded in encrypted form into the
     * authorization code and/or access/refresh tokens to enable recovery on the back-channel.
     *
     * @param strategy  lookup strategy
     */
    public void setEncodedAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        encodedAttributesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the set of attribute IDs which should be omitted from the UserInfo token.
     * 
     * <p>Default behavior is to include all claims, but omiited claims also affect the set that
     * may need to be embedded for recovery into the access/refresh tokens.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the attribute IDs to omit from UserInfo token
     */
    @Nonnull @NonnullElements @NotLive public Set<String> getDeniedUserInfoAttributes(
            @Nullable final ProfileRequestContext profileRequestContext) {
        
        final Set<String> attributes = deniedUserInfoAttributesLookupStrategy.apply(profileRequestContext);
        if (attributes != null) {
            return Set.copyOf(attributes);
        }
        return Collections.emptySet();
    }

    /**
     * Set the set of attribute IDs which should be omitted from the UserInfo token.
     * 
     * <p>Default behavior is to include all claims, but omiited claims also affect the set that
     * may need to be embedded for recovery into the access/refresh tokens.</p>
     * 
     * @param attributes the attribute IDs to omit from UserInfo token
     */
    public void setDeniedUserInfoAttributes(@Nullable @NonnullElements final Collection<String> attributes) {

        if (attributes == null || attributes.isEmpty()) {
            deniedUserInfoAttributesLookupStrategy = FunctionSupport.constant(null);
        } else {
            deniedUserInfoAttributesLookupStrategy = FunctionSupport.constant(
                    Set.copyOf(StringSupport.normalizeStringCollection(attributes)));
        }
    }

    /**
     * Set a lookup strategy for the set of attribute IDs which should be omitted from the UserInfo token.
     *
     * @param strategy  lookup strategy
     */
    public void setDeniedUserInfoAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        deniedUserInfoAttributesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get whether to include iss parameter in the authentication response.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return whether to include iss parameter in the authentication response
     * 
     * @since 2.1.0
     */
    public boolean isIncludeIssuerInResponse(@Nullable final ProfileRequestContext profileRequestContext) {
        return includeIssuerInResponsePredicate.test(profileRequestContext);
    }

    /**
     * Set whether to include iss parameter in the authentication response.
     * 
     * @param flag flag to set
     * 
     * @since 2.1.0
     */
    public void setIncludeIssuerInResponse(final boolean flag) {
        includeIssuerInResponsePredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set condition for whether to include iss parameter in the authentication response.
     * 
     * @param condition condition to set
     * 
     * @since 2.1.0
     */
    public void setIncludeIssuerInResponsePredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        includeIssuerInResponsePredicate = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Get the bi-function for manipulating authorization code claims set.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the bi-function for manipulating authorization code claims set
     * 
     * @since 2.1.0
     */
    @Nonnull
    public BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>
        getAuthorizationCodeClaimsSetManipulationStrategy(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return authorizationCodeClaimsSetManipulationStrategyLookupStrategy.apply(profileRequestContext);
    }

    /**
     * Set the bi-function for manipulating authorization code claims set.
     * 
     * @param strategy bi-function for manipulating authorization code claims set
     * 
     * @since 2.1.0
     */
    public void setAuthorizationCodeClaimsSetManipulationStrategy(
            @Nullable final BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> strategy) {
        authorizationCodeClaimsSetManipulationStrategyLookupStrategy = FunctionSupport.constant(strategy);
    }

    /**
     * Set a lookup strategy for the bi-function for manipulating authorization code claims set.
     *
     * @param strategy lookup strategy
     * 
     * @since 2.1.0
     */
    public void setAuthorizationCodeClaimsSetManipulationStrategyLookupStrategy(@Nonnull final 
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        authorizationCodeClaimsSetManipulationStrategyLookupStrategy = Constraint.isNotNull(strategy,
                "Lookup strategy cannot be null");
    }

}