/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.profile.config;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import com.google.common.base.Predicates;

import net.shibboleth.idp.authn.config.AuthenticationProfileConfiguration;
import net.shibboleth.idp.profile.config.AttributeResolvingProfileConfiguration;
import net.shibboleth.idp.profile.config.OverriddenIssuerProfileConfiguration;
import net.shibboleth.oidc.profile.oauth2.config.AbstractOAuth2FlowAwareProfileConfiguration;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Base class for profiles that support OIDC's concept of SSO, which spans multiple endpoints.
 */
public abstract class AbstractOIDCSSOConfiguration extends AbstractOAuth2FlowAwareProfileConfiguration
        implements OIDCProfileConfiguration, AuthenticationProfileConfiguration, OverriddenIssuerProfileConfiguration,
            AttributeResolvingProfileConfiguration {

    /** OIDC base protocol URI. */
    @Nonnull @NotEmpty public static final String PROTOCOL_URI = "http://openid.net/specs/openid-connect-core-1_0.html";

    /** Lookup function to override issuer value. */
    @Nonnull private Function<ProfileRequestContext,String> issuerLookupStrategy;

    /** Whether attributes should be resolved in the course of the profile. */
    @Nonnull private Predicate<ProfileRequestContext> resolveAttributesPredicate;

    /** Whether encryption is optional in the face of no key, etc. */
    @Nonnull private Predicate<ProfileRequestContext> encryptionOptionalPredicate;
    
    /** Whether client is required to use PKCE. */
    @Nonnull private Predicate<ProfileRequestContext> forcePKCEPredicate;

    /** Whether client is allowed to use PKCE code challenge method plain. */
    @Nonnull private Predicate<ProfileRequestContext> allowPKCEPlainPredicate;
    
    /** Lookup function to supply ID token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> idTokenLifetimeLookupStrategy;

    /** Lookup functioon to supply access token type. */
    @Nonnull private Function<ProfileRequestContext,String> accessTokenTypeLookupStrategy;
    
    /** Lookup function to supply access token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> accessTokenLifetimeLookupStrategy;
    
    /** Lookup function to supply refresh token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> refreshTokenLifetimeLookupStrategy;
    
    /** Lookup function to supply additional audiences for ID token. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> assertionAudiencesLookupStrategy;

    /** Lookup function to supply attribute IDs to include in ID token regardless of response_type. */
    @Nonnull private Function<ProfileRequestContext,Set<String>> alwaysIncludedAttributesLookupStrategy;

    /** Lookup function to supply strategy bi-function for manipulating id_token claims. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        idTokenManipulationStrategyLookupStrategy;

    /** Lookup function to supply strategy bi-function for manipulating access token claims set. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        accessTokenClaimsSetManipulationStrategyLookupStrategy;

    /**
     * Creates a new configuration instance.
     *
     * @param profileId Unique profile identifier.
     */
    public AbstractOIDCSSOConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
        issuerLookupStrategy = FunctionSupport.constant(null);
        resolveAttributesPredicate = Predicates.alwaysTrue();
        encryptionOptionalPredicate = Predicates.alwaysTrue();

        forcePKCEPredicate = Predicates.alwaysFalse();
        allowPKCEPlainPredicate = Predicates.alwaysFalse();
        
        idTokenLifetimeLookupStrategy = FunctionSupport.constant(Duration.ofHours(1));
        accessTokenTypeLookupStrategy = FunctionSupport.constant(null);
        accessTokenLifetimeLookupStrategy = FunctionSupport.constant(Duration.ofMinutes(10));
        refreshTokenLifetimeLookupStrategy = FunctionSupport.constant(Duration.ofHours(2));
        
        assertionAudiencesLookupStrategy = FunctionSupport.constant(null);
        alwaysIncludedAttributesLookupStrategy = FunctionSupport.constant(null);

        idTokenManipulationStrategyLookupStrategy = FunctionSupport.constant(null);
        accessTokenClaimsSetManipulationStrategyLookupStrategy = FunctionSupport.constant(null);
    }
    
    /** {@inheritDoc} */
    @Nullable @NotEmpty public String getIssuer(@Nullable final ProfileRequestContext profileRequestContext) {
        return issuerLookupStrategy.apply(profileRequestContext);
    }
    
    /**
     * Set overridden issuer value.
     * 
     * @param issuer issuer value
     */
    public void setIssuer(@Nullable @NotEmpty final String issuer) {
        issuerLookupStrategy = FunctionSupport.constant(issuer);
    }
    
    /**
     * Sets lookup strategy for overridden issuer value.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
        issuerLookupStrategy = Constraint.isNotNull(strategy, "Issuer lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    public boolean isResolveAttributes(@Nullable final ProfileRequestContext profileRequestContext) {
        return resolveAttributesPredicate.test(profileRequestContext);
    }

    /**
     * Set whether attributes should be resolved during the profile.
     * 
     * @param flag flag to set
     */
    public void setResolveAttributes(final boolean flag) {
        resolveAttributesPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }
    
    /**
     * Set a condition to determine whether attributes should be resolved during the profile.
     * 
     * @param condition condition to set
     */
    public void setResolveAttributesPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        resolveAttributesPredicate = Constraint.isNotNull(condition, "Resolve attributes predicate cannot be null");
    }

    /**
     * Get whether encryption is optional in the face of a missing key, etc.
     * 
     * @param profileRequestContext current profile request context
     * 
     * @return true iff encryption is optional
     */
    public boolean isEncryptionOptional(@Nullable final ProfileRequestContext profileRequestContext) {
        return encryptionOptionalPredicate.test(profileRequestContext);
    }
    
    /**
     * Set whether encryption is optional in the face of a missing key, etc.
     * 
     * @param flag  flag to set
     */
    public void setEncryptionOptional(final boolean flag) {
        encryptionOptionalPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

    /**
     * Set a condition to determine whether encryption is optional in the face of a missing key, etc.
     *
     * @param condition condition to set
     */
    public void setEncryptionOptionalPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        encryptionOptionalPredicate = Constraint.isNotNull(condition, "Encryption optional predicate cannot be null");
    }

   /**
    * Get whether client is required to use PKCE.
    * 
    * @param profileRequestContext profile request context
    * 
    * @return whether client is required to use PKCE
    */
    public boolean isForcePKCE(@Nullable final ProfileRequestContext profileRequestContext) {
        return forcePKCEPredicate.test(profileRequestContext);
    }

   /**
    * Set whether client is required to use PKCE.
    * 
    * @param flag flag to set
    */
    public void setForcePKCE(final boolean flag) {
        forcePKCEPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

   /**
    * Set condition for whether client is required to use PKCE.
    * 
    * @param condition condition to set
    */
    public void setForcePKCEPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        forcePKCEPredicate = Constraint.isNotNull(condition, "Condition cannot be null");
    }

   /**
    * Get whether client is allowed to use PKCE code challenge method plain.
    * 
    * @param profileRequestContext profile request context
    * 
    * @return whether client is allowed to use PKCE code challenge method plain
    */
    public boolean isAllowPKCEPlain(@Nullable final ProfileRequestContext profileRequestContext) {
        return allowPKCEPlainPredicate.test(profileRequestContext);
    }

   /**
    * Set whether client is allowed to use PKCE code challenge method plain.
    * 
    * @param flag flag to set
    */
    public void setAllowPKCEPlain(final boolean flag) {
        allowPKCEPlainPredicate = flag ? Predicates.alwaysTrue() : Predicates.alwaysFalse();
    }

   /**
    * Set condition for whether client is allowed to use PKCE code challenge method plain.
    * 
    * @param condition condition to set
    */
    public void setAllowPKCEPlainPredicate(@Nonnull final Predicate<ProfileRequestContext> condition) {
        allowPKCEPlainPredicate = Constraint.isNotNull(condition, "Condition cannot be null");
    }

   /**
    
    /**
     * Get ID token lifetime.
     * 
     * <p>Defaults to 1 hour.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return ID token lifetime
     */
    @Positive @Nonnull public Duration getIDTokenLifetime(@Nullable final ProfileRequestContext profileRequestContext) {
        
        final Duration lifetime = idTokenLifetimeLookupStrategy.apply(profileRequestContext);
        
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "ID token lifetime must be greater than 0");
        return lifetime;
    }

    /**
     * Set the lifetime of an id token.
     * 
     * @param lifetime lifetime of an id token in milliseconds
     */
    public void setIDTokenLifetime(@Positive @Nonnull final Duration lifetime) {
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "ID token lifetime must be greater than 0");
        
        idTokenLifetimeLookupStrategy = FunctionSupport.constant(lifetime);
    }

    /**
     * Set a lookup strategy for the ID token lifetime.
     *
     * @param strategy lookup strategy
     */
    public void setIDTokenLifetimeLookupStrategy(@Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        idTokenLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }
    
    /**
     * Get access token type.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return access token type, or null for unspecified/opaque
     * 
     * @since 2.1.0
     */
     @Nullable @NotEmpty public String getAccessTokenType(@Nullable final ProfileRequestContext profileRequestContext) {
         return accessTokenTypeLookupStrategy.apply(profileRequestContext);
     }
    
    /**
     * Set access token type.
     * 
     * @param type token type, or null for unspecified/opaque
     * 
     * @since 2.1.0
     */
     public void setAccessTokenType(@Nullable @NotEmpty final String type) {
         accessTokenTypeLookupStrategy = FunctionSupport.constant(StringSupport.trimOrNull(type));
     }
    
    /**
     * Set lookup strategy for access token type.
     * 
     * @param strategy lookup strategy
     * 
     * @since 2.1.0
     */
     public void setAccessTokenTypeLookupStrategy(@Nonnull final Function<ProfileRequestContext,String> strategy) {
         accessTokenTypeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
     }

    /**
     * Get access token lifetime.
     * 
     * <p>Defaults to 10 minutes.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return access token lifetime
     */
    @Positive @Nonnull
    public Duration getAccessTokenLifetime(@Nullable final ProfileRequestContext profileRequestContext) {
        final Duration lifetime = accessTokenLifetimeLookupStrategy.apply(profileRequestContext);
        
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Access token lifetime must be greater than 0");
        return lifetime;
    }
    
    /**
     * Set the lifetime of an access token.
     * 
     * @param lifetime lifetime of an access token in milliseconds
     */
    public void setAccessTokenLifetime(@Positive @Nonnull final Duration lifetime) {
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Access token lifetime must be greater than 0");
        
        accessTokenLifetimeLookupStrategy = FunctionSupport.constant(lifetime);
    }
    
    /**
     * Set a lookup strategy for the access token lifetime.
     *
     * @param strategy lookup strategy
     */
    public void setAccessTokenLifetimeLookupStrategy(
            @Nullable final Function<ProfileRequestContext,Duration> strategy) {
        accessTokenLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }
    
    /**
     * Get refresh token lifetime.
     * 
     * <p>Defaults to 2 hours.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return refresh token lifetime
     */
    @Nonnull @Positive
    public Duration getRefreshTokenLifetime(@Nullable final ProfileRequestContext profileRequestContext) {
        final Duration lifetime = refreshTokenLifetimeLookupStrategy.apply(profileRequestContext);
        
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Refresh token lifetime must be greater than 0");
        return lifetime;
    }

    /**
     * Set the lifetime of refresh token.
     * 
     * @param lifetime lifetime of an refresh token
     */
    public void setRefreshTokenLifetime(@Nonnull @Positive final Duration lifetime) {
        Constraint.isTrue(lifetime != null && !lifetime.isZero() && !lifetime.isNegative(),
                "Refresh token lifetime must be greater than 0");
        
        refreshTokenLifetimeLookupStrategy = FunctionSupport.constant(lifetime);
    }

    /**
     * Set a lookup strategy for the refresh token lifetime.
     *
     * @param strategy lookup strategy
     */
    public void setRefreshTokenLifetimeLookupStrategy(
            @Nullable final Function<ProfileRequestContext,Duration> strategy) {
        refreshTokenLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the set of audiences, in addition to the relying party(ies) to which the IdP is issuing the ID Token, with
     * which the token may be shared.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the additional audiences
     */
    @Nonnull @NonnullElements @NotLive public Set<String> getAdditionalAudiencesForIdToken(
            @Nullable final ProfileRequestContext profileRequestContext) {
        
        final Set<String> audiences = assertionAudiencesLookupStrategy.apply(profileRequestContext);
        if (audiences != null) {
            return Set.copyOf(audiences);
        }
        return Collections.emptySet();
    }

    /**
     * Set the set of audiences, in addition to the relying party(ies) to which the IdP is issuing the ID Token, with
     * which the token may be shared.
     * 
     * @param audiences the additional audiences
     */
    public void setAdditionalAudiencesForIdToken(@Nullable @NonnullElements final Collection<String> audiences) {

        if (audiences == null || audiences.isEmpty()) {
            assertionAudiencesLookupStrategy = FunctionSupport.constant(null);
        } else {
            assertionAudiencesLookupStrategy = FunctionSupport.constant(
                    Set.copyOf(StringSupport.normalizeStringCollection(audiences)));
        }
    }

    /**
     * Set a lookup strategy for the set of audiences, in addition to the relying party(ies) to which the IdP
     * is issuing the ID Token, with which the token may be shared.
     *
     * @param strategy  lookup strategy
     */
    public void setAdditionalAudiencesForIdTokenLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        assertionAudiencesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the set of attribute IDs which should be included in the ID token regardless of response_type.
     * 
     * <p>Default behavior is to include claims only with the implicit id_token type only, while any use
     * of a back-channel relies on the user_info endpoint to get the claims. This setting forces certain
     * attributes to be added to the ID token regardless of flow.</p>
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the attribute IDs to include in all cases
     */
    @Nonnull @NonnullElements @NotLive public Set<String> getAlwaysIncludedAttributes(
            @Nullable final ProfileRequestContext profileRequestContext) {
        
        final Set<String> attributes = alwaysIncludedAttributesLookupStrategy.apply(profileRequestContext);
        if (attributes != null) {
            return Set.copyOf(attributes);
        }
        return Collections.emptySet();
    }

    /**
     * Set the set of attribute IDs which should be included in the ID token regardless of response_type.
     * 
     * <p>Default behavior is to include claims only with the implicit id_token type only, while any use
     * of a back-channel relies on the user_info endpoint to get the claims. This setting forces certain
     * attributes to be added to the ID token regardless of flow.</p>
     * 
     * @param attributes the attribute IDs to include in all cases
     */
    public void setAlwaysIncludedAttributes(@Nullable @NonnullElements final Collection<String> attributes) {

        if (attributes == null || attributes.isEmpty()) {
            alwaysIncludedAttributesLookupStrategy = FunctionSupport.constant(null);
        } else {
            alwaysIncludedAttributesLookupStrategy = FunctionSupport.constant(
                    Set.copyOf(StringSupport.normalizeStringCollection(attributes)));
        }
    }

    /**
     * Set a lookup strategy for the attribute IDs which should be included in the ID token regardless of
     * response_type.
     *
     * @param strategy  lookup strategy
     */
    public void setAlwaysIncludedAttributesLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Set<String>> strategy) {
        alwaysIncludedAttributesLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the bi-function for manipulating id_token claims.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the bi-function for manipulating id_token claims
     * 
     * @since 2.1.0
     */
    @Nonnull
    public BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> getIDTokenManipulationStrategy(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return idTokenManipulationStrategyLookupStrategy.apply(profileRequestContext);
    }

    /**
     * Set the bi-function for manipulating id_token claims.
     * 
     * @param strategy bi-function for manipulating id_token claims
     * 
     * @since 2.1.0
     */
    public void setIDTokenManipulationStrategy(
            @Nullable final BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> strategy) {
        idTokenManipulationStrategyLookupStrategy = FunctionSupport.constant(strategy);
    }

    /**
     * Set a lookup strategy for the bi-function for manipulating id_token claims.
     *
     * @param strategy lookup strategy
     * 
     * @since 2.1.0
     */
    public void setIDTokenManipulationStrategyLookupStrategy(@Nonnull final 
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        idTokenManipulationStrategyLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /**
     * Get the bi-function for manipulating access token claims set.
     * 
     * @param profileRequestContext profile request context
     * 
     * @return the bi-function for manipulating access token claims set
     * 
     * @since 2.1.0
     */
    @Nonnull
    public BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>
        getAccessTokenClaimsSetManipulationStrategy(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return accessTokenClaimsSetManipulationStrategyLookupStrategy.apply(profileRequestContext);
    }

    /**
     * Set the bi-function for manipulating access token claims set.
     * 
     * @param strategy bi-function for manipulating access token claims set
     * 
     * @since 2.1.0
     */
    public void setAccessTokenClaimsSetManipulationStrategy(
            @Nullable final BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> strategy) {
        accessTokenClaimsSetManipulationStrategyLookupStrategy = FunctionSupport.constant(strategy);
    }

    /**
     * Set a lookup strategy for the bi-function for manipulating access token claims set.
     *
     * @param strategy lookup strategy
     * 
     * @since 2.1.0
     */
    public void setAccessTokenClaimsSetManipulationStrategyLookupStrategy(@Nonnull final 
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        accessTokenClaimsSetManipulationStrategyLookupStrategy = Constraint.isNotNull(strategy,
                "Lookup strategy cannot be null");
    }

}