/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.policy.impl;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.FileSystemResource;

import net.shibboleth.oidc.metadata.cache.CacheLoadingContext;
import net.shibboleth.oidc.metadata.cache.CacheLoadingException;
import net.shibboleth.oidc.metadata.cache.LoadingStrategy;
import net.shibboleth.oidc.metadata.cache.impl.DefaultFileLoadingStrategy;
import net.shibboleth.oidc.metadata.criterion.ResourceLocationCriterion;
import net.shibboleth.oidc.metadata.impl.AbstractDynamicHTTPFetchingStrategy;
import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A fetching strategy that exploits {@link ResourceLocationCriterion} from the given {@link CriteriaSet} to resolve
 * a location for metadata policy. If the value starts with <code>http://</code> or <code>https://</code>, the policy
 * is fetched via {@link HttpClient}. In other cases the location is expected to be a file path. Possibly existing
 * <code>file:</code> prefix is removed before loading the file.
 */
public class MetadataPolicyViaLocationFetchingStrategy
        extends AbstractDynamicHTTPFetchingStrategy<Map<String, MetadataPolicy>>
        implements Function<CriteriaSet, Map<String, MetadataPolicy>> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(MetadataPolicyViaLocationFetchingStrategy.class);
    
    /** The parsing strategy used for parsing metadata policies. */
    @NonnullAfterInit private Function<byte[], List<Map<String, MetadataPolicy>>> parsingStrategy;

    /**
     * Constructor.
     *
     * @param client the instance of {@link HttpClient} used to fetch remote metadata policy.
     * @param handler the response handler used to convert the HTTP response to the metadata policy.
     */
    public MetadataPolicyViaLocationFetchingStrategy(final HttpClient client,
            final ResponseHandler<Map<String, MetadataPolicy>> handler) {
        super(client, handler);
    }
    
    /**
     * Set the parsing strategy used for parsing metadata policies.
     * 
     * @param strategy What to set.
     */
    public void setParsingStrategy(@Nonnull final Function<byte[], List<Map<String, MetadataPolicy>>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        parsingStrategy = Constraint.isNotNull(strategy, "Parsing strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (parsingStrategy == null) {
            throw new ComponentInitializationException("Parsing strategy cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Map<String, MetadataPolicy> apply(final @Nonnull CriteriaSet criteriaSet) {
        final String requestURL = buildRequestURL(criteriaSet);
        if (requestURL == null) {
            return null;
        }
        if (requestURL.startsWith("http://") || requestURL.startsWith("https://")) {
            return super.apply(criteriaSet);
        }
        final String fileLocation = requestURL.startsWith("file:") ? requestURL.substring(5) : requestURL;
        final CacheLoadingContext context = new CacheLoadingContext(null, null);
        final FileSystemResource resource = new FileSystemResource(fileLocation);
        final byte[] loadedPolicy;
        try {
            final LoadingStrategy loadingStrategy = new DefaultFileLoadingStrategy(resource);
            loadedPolicy = loadingStrategy.load(context);
            if (loadedPolicy == null) {
                log.error("Could not load any entries via loading strategy");
                return null;
            }
        } catch (final IOException | CacheLoadingException e) {
            log.error("Could not load a metadata policy from file {}", fileLocation, e);
            return null;
        }        
        final List<Map<String, MetadataPolicy>> result = parsingStrategy.apply(loadedPolicy);
        if (result == null || result.isEmpty()) {
            log.warn("Could not find any entries via parsing strategy");
            return null;
        }
        if (result.size() > 1) {
            log.warn("More than one metadata policies found, returning first of the list.");
        }
        return result.get(0);
    }

    /** {@inheritDoc} */
    @Override
    protected String buildRequestURL(final CriteriaSet criteria) {
        if (criteria == null || criteria.isEmpty()) {
            return null;
        }
        final ResourceLocationCriterion criterion = criteria.get(ResourceLocationCriterion.class);
        return criterion == null ? null : criterion.getResourceLocation();
    }

}
