/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.policy.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;

/**
 * Static utility methods related to metadata policies.
 */
public final class MetadataPolicyHelper {
    
    /**
     * Constructor.
     */
    private MetadataPolicyHelper() {
        // no op
    }
    
    /**
     * Checks if the given candidate is a superset of the given values.
     * 
     * @param candidate The candidate to be checked. May not be null.
     * @param values The values to be checked. May not be null.
     * @return true if the candidate is a superset of the values or they are equal, false otherwise.
     */
    public static boolean isSupersetOfValues(@Nonnull final Object candidate, @Nonnull final Collection<?> values) {
        return candidate instanceof Collection
                ? ((Collection<?>) candidate).containsAll(values)
                : values.size() == 1 && values.contains(candidate);
    }
    
    /**
     * Checks if the given candiate is a subset of the given values.
     * 
     * @param candidate The candidate to be checked. May not be null.
     * @param values The values to be checked. May not be null.
     * @return true if the candidate is a subset of the values or they are equal, false otherwise.
     */
    public static boolean isSubsetOfValues(@Nonnull final Object candidate, @Nonnull final Collection<?> values) {
        return candidate instanceof Collection
                ? values.containsAll((Collection<?>) candidate)
                : values.contains(candidate);
    }
    
 // Checkstyle: CyclomaticComplexity OFF
    /**
     * <p>Merges two metadata policies with the rules defined in the OIDC federation spec 5.1.3.1:</p>
     * <ul>
     * <li>
     * subset_of: The result of merging the values of two subset_of operators is the intersection of the operator
     * values.
     * </li>
     * <li>
     * one_of: The result of merging the values of two one_of operators is the intersection of the operator values.
     * </li>
     * <li>
     * superset_of: The result of merging the values of two superset_of operators is the union of the operator values.
     * </li>
     * <li>
     * add: The result of merging the values of two add operators is the union of the values.
     * </li>
     * <li>
     * value: Merging two value operators is NOT allowed unless the two operator values are equal.
     * </li>
     * <li>
     * default: Merging two default operators is NOT allowed unless the two operator values are equal.
     * </li>
     * <li>
     * essential: If a superior has specified essential=true, then a subordinate cannot change that. If a superior has
     * specified essential=false, then a subordinate is allowed to change that to essential=true. If a superior has not
     * specified essential, then a subordinate can set essential to true or false.
     * </li>
     * </ul>
     * 
     * <p>In addition to the list above, 'regex' operator is treated in the same way as 'value' and 'default'.</p>
     * 
     * @param superior The superior metadata policy.
     * @param subordinate The subordinate metadata policy.
     * @return The merged metadata policy.
     * @throws ConstraintViolationException If two 'value' or 'default' operators with different values are attempted
     * to be merged.
     */
    @Nullable public static MetadataPolicy mergeMetadataPolicies(@Nullable final MetadataPolicy superior,
            @Nullable final MetadataPolicy subordinate) throws ConstraintViolationException {
        if (superior == null) {
            return subordinate;
        }
        if (subordinate == null) {
            return superior;
        }
        final Object defaultValue = superior.getDefaultValue();
        if (defaultValue != null) {
            final Object anotherDefault = subordinate.getDefaultValue();
            if (anotherDefault != null && !defaultValue.equals(anotherDefault)) {
                throw new ConstraintViolationException(
                        "Merging two default operators is NOT allowed unless the two operator values are equal.");
            }
        }
        final Object value = superior.getValue();
        if (value != null) {
            final Object anotherValue = subordinate.getValue();
            if (anotherValue != null && !value.equals(anotherValue)) {
                throw new ConstraintViolationException(
                        "Merging two value operators is NOT allowed unless the two operator values are equal.");
                
            }
        }
        final String regexp = superior.getRegexp();
        if (regexp != null) {
            final String anotherRegexp = subordinate.getRegexp();
            if (anotherRegexp != null && !regexp.equals(anotherRegexp)) {
                throw new ConstraintViolationException(
                        "Mering two regexp operators is NOT allowed unless the two operator values are equal.");
            }
        }
        return new MetadataPolicy.Builder()
                .withSubsetOfValues(doMergeForTwoLists(superior.getSubsetOfValues(),
                        subordinate.getSubsetOfValues(), false))
                .withOneOfValues(doMergeForTwoLists(superior.getOneOfValues(), subordinate.getOneOfValues(), false))
                .withSupersetOfValues(doMergeForTwoLists(superior.getSupersetOfValues(),
                        subordinate.getSupersetOfValues(), true))
                .withAdd(doMergeForTwoObjects(superior.getAdd(), subordinate.getAdd(), true))
                .withValue(value)
                .withDefaultValue(defaultValue)
                .withEssential(superior.isEssential() ? Boolean.TRUE : subordinate.getEssential())
                .withRegexp(regexp)
                .build();
    }
 // Checkstyle: CyclomaticComplexity ON
    /**
     * Merges two metadata policy values that can be either single valued or lists.
     * 
     * @param superior The superior metadata value.
     * @param subordinate The subordinate metadata value.
     * @param union Flag to indicate the use of union. Intersection is used if false.
     * @return The merged value.
     * @throws ConstraintViolationException If the values are conflicting: two different values cannot be
     * merged when union-flag is true.
     */
    @Nullable private static Object doMergeForTwoObjects(@Nullable final Object superior,
            @Nullable final Object subordinate, final boolean union) throws ConstraintViolationException {
        if (superior == null) {
            return subordinate;
        }
        // check if either of superior or subordinate are lists and process if they are
        if (superior instanceof List) {
            if (subordinate instanceof List) {
                return doMergeForTwoLists(new ArrayList<Object>((List<?>) superior), (List<?>) subordinate, union);
            }
            return doMergeForTwoLists(new ArrayList<Object>((List<?>) superior),
                    subordinate == null ? null : List.of(subordinate), union);
        } else if (subordinate instanceof List) {
            return doMergeForTwoLists(List.of(superior), (List<?>) subordinate, union);
        }
        // superior and subordinate are plain objects
        if (subordinate == null) {
            return superior;
        }
        if (union) {
            if (superior.equals(subordinate)) {
                return superior;
            } else {
                return List.of(superior, subordinate);
            }
        }
        if (!superior.equals(subordinate)) {
            throw new ConstraintViolationException("Merging of two different values is not allowed");
        }
        return superior;
    }

    /**
     * Merges two metadata policy list values.
     * 
     * @param superior The superior value.
     * @param subordinate The subordinate value.
     * @param union Flag to indicate the use of union. Intersection is used if false.
     * @return The merged value.
     */
    @SuppressWarnings("unchecked")
    @Nullable private static List<Object> doMergeForTwoLists(@Nullable final List<?> superior,
            @Nullable final List<?> subordinate, final boolean union) {
        if (superior == null) {
            return (List<Object>) subordinate;
        }
        if (subordinate == null) {
            return List.copyOf(superior);
        }
        if (union) {
            final Set<Object> set = new HashSet<>(superior);
            set.addAll(subordinate);
            return List.copyOf(set);
        }
        final List<Object> result = new ArrayList<>(superior);
        result.retainAll(subordinate);
        return result;
    }
}