/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.policy.impl;

import java.util.List;
import java.util.Map;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.utilities.java.support.collection.Pair;

/**
 * <p>
 * A function that verifies that the map of {@link MetadataPolicy} entries meets the restrictions defined in the
 * OIDC federation specification 1.0 (draft 17 / September 2021):
 * </p>
 * 
 * <p>
 * A policy entry can contain one or more operators. Not all operators are allowed to appear together in a policy
 * entry.
 * </p>
 * 
 * <ul>
 * <li>subset_of and superset_of applies to parameters that can have more than one value (for instance, contacts)
 * while one_of applies to parameters that can only have one value (for instance, id_token_signed_response_alg). This
 * means that one_of cannot appear beside subset_of/ superset_of in a policy entry.</li>
 * <li>value overrides everything else. So having value together with any other operator (except for essential) does
 * not make sense.</li>
 * <li>If subset_of and superset_of both appear as operators, then the list of values in subset_of MUST be a superset
 * of the values in superset_of.<li>
 * <li>If add appears in a policy entry together with subset_of then the value/values of add MUST be a subset of
 * subset_of.<li>
 * <li>If add appears in a policy entry together with superset_of then the values of add MUST be a superset of
 * superset_of.<li>
 * <li>If default appears in a policy entry together with subset_of then the values of default MUST be a subset of
 * subset_of.<li>
 * <li>If default appears in a policy entry together with superset_of then the values of default MUST be a superset of
 * superset_of.<li>
 * <li>If add appears in a policy entry together with one_of then the value of add MUST be a member of one_of.<li>
 * <li>If default appears in a policy entry together with one_of then the value default MUST be a member of one_of.
 * </li>
 * </ul>
 */
public class DefaultMetadataPolicyValidator implements Predicate<Map<String, MetadataPolicy>> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultMetadataPolicyValidator.class);
    

    /** {@inheritDoc} */
 // Checkstyle: CyclomaticComplexity OFF
    @Override
    public boolean test(@Nullable final Map<String, MetadataPolicy> map) {
        boolean result = true;
        if (map == null || map.isEmpty()) {
            return true;
        }
        for (final String claim : map.keySet()) {
            final MetadataPolicy policy = map.get(claim);
            final List<Object> oneOfValues = policy.getOneOfValues();
            final List<Object> subsetOfValues = policy.getSubsetOfValues();
            final List<Object> supersetOfValues = policy.getSupersetOfValues();
            if ((subsetOfValues != null || supersetOfValues != null) && oneOfValues != null) {
                log.warn("Claim {}: one_of cannot be set when superset_of and/or subset_of is set", claim);
                result = false;
            }

            final Object value = policy.getValue();
            final Object add = policy.getAdd();
            final Object defaultValue = policy.getDefaultValue();
            final String regexp = policy.getRegexp();

            if (value != null) {
                if (add != null || defaultValue != null || oneOfValues != null || regexp != null
                        || subsetOfValues != null || supersetOfValues != null) {
                    log.warn("Claim {}: value is set together with modifiers or value checks (other than essential)",
                            claim);
                    result = false;
                }
            }
            if (supersetOfValues != null && subsetOfValues != null) {
                if (!subsetOfValues.containsAll(supersetOfValues)) {
                    log.warn("Claim {}: subset_of and superset_of set, but subset_of is not a superset of superset_of",
                            claim);
                    result = false;
                }
            }
            
            if (!verifyValue(claim, new Pair<>("add", add), subsetOfValues, supersetOfValues, oneOfValues)) {
                result = false;
            }

            if (!verifyValue(claim, new Pair<>("default", defaultValue), subsetOfValues, supersetOfValues,
                    oneOfValues)) {
                result = false;
            }

        }
        return result;
    }
 // Checkstyle: CyclomaticComplexity ON
    
    /**
     * <p>
     * Verifies the value against the following rules:
     * </p>
     * 
     * <ul>
     * <li>If it appears in a policy entry together with subset_of then the value/values of add MUST be a subset of
     * subset_of.</li>
     * <li>If it appears in a policy entry together with superset_of then the values of add MUST be a superset of
     * superset_of.</li>
     * <li>If it appears in a policy entry together with one_of then the value of add MUST be a member of one_of.</li>
     * </ul>
     * 
     * @param claim The claim whose metadata policy is being verified, used in logging.
     * @param value The value to be verified against the rules.
     * @param subsetOfValues The contents of subset_of.
     * @param supersetOfValues The contents of superset_of.
     * @param oneOfValues The contents of one_of.
     * @return true if the value meets the rules, false otherwise.
     */
    protected boolean verifyValue(final String claim, final Pair<String, Object> value,
            final List<Object> subsetOfValues, final List<Object> supersetOfValues, final List<Object> oneOfValues) {
        boolean result = true;
        final String valueId = value.getFirst();
        final Object valueObject = value.getSecond();
        if (valueObject != null) {
            if (subsetOfValues != null && !MetadataPolicyHelper.isSubsetOfValues(valueObject, subsetOfValues)) {
                log.warn("Claim {}: {} contains values that are not subset of values in subset_of", claim, valueId);
                result = false;
            }
            if (supersetOfValues != null && !MetadataPolicyHelper.isSupersetOfValues(valueObject, supersetOfValues)) {
                log.warn("Claim {}: {} contains values that are not superset of values in superset_of", claim,
                        valueId);
                result = false;
            }
            if (oneOfValues != null && !oneOfValues.contains(valueObject)) {
                log.warn("Claim {}: {} contains value that is not included in one_of", claim, valueId);
                result = false;
            }
        }
        return result;
    }

}
