/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.keyinfo.ext.impl.provider;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.security.credential.Credential;
import org.opensaml.xmlsec.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xmlsec.keyinfo.impl.KeyInfoResolutionContext;
import org.opensaml.xmlsec.keyinfo.impl.provider.AbstractKeyInfoProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.auth.Secret;

import net.shibboleth.oidc.saml.xmlobject.ClientSecret;
import net.shibboleth.oidc.security.credential.BasicNimbusSecretCredential;
import net.shibboleth.oidc.security.credential.NimbusSecretCredential;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * Implementation of {@link org.opensaml.xmlsec.keyinfo.impl.KeyInfoProvider} which provides basic support for
 * extracting a {@link NimbusSecretCredential} child of KeyInfo.
 */
public class ClientSecretProvider extends AbstractKeyInfoProvider {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(ClientSecretProvider.class);

    /** {@inheritDoc} */
    @Override @Nullable
    public Collection<Credential> process(@Nonnull final KeyInfoCredentialResolver resolver,
            @Nonnull final XMLObject keyInfoChild, @Nullable final CriteriaSet criteriaSet,
            @Nonnull final KeyInfoResolutionContext kiContext) throws SecurityException {
        final ClientSecret clientSecret = getClientSecret(keyInfoChild);
        if (clientSecret == null || StringSupport.trimOrNull(clientSecret.getValue()) == null) {
            log.debug("No value found for the ClientSecret element");
            return null;
        }
        final LazySet<Credential> credentials = new LazySet<>();
        credentials.add(new BasicNimbusSecretCredential(new Secret(clientSecret.getValue())));
        return credentials;
    }

    /** {@inheritDoc} */
    @Override
    public boolean handles(@Nonnull final XMLObject keyInfoChild) {
        return getClientSecret(keyInfoChild) != null;
    }

    /**
     * Get the ClientSecret from the passed XML object.
     * 
     * @param xmlObject an XML object, presumably a {@link ClientSecret}
     * @return the ClientSecret which was found, or null if none
     */
    @Nullable protected ClientSecret getClientSecret(@Nonnull final XMLObject xmlObject) {

        if (xmlObject instanceof ClientSecret) {
            return (ClientSecret) xmlObject;
        } else {
            return null;
        }
    }
}
