/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.io.IOException;
import java.time.Instant;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.storage.StorageService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

import net.shibboleth.oidc.metadata.ClientInformationManager;
import net.shibboleth.oidc.metadata.ClientInformationManagerException;

/**
 * A {@link ClientInformationManager} exploiting {@link StorageService} for storing the data.
 */
public class StorageServiceClientInformationManager extends BaseStorageServiceClientInformationComponent 
    implements ClientInformationManager {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(StorageServiceClientInformationResolver.class);
    
    /** {@inheritDoc} */
    @Override
    public void storeClientInformation(@Nonnull final OIDCClientInformation clientInformation,
            @Nullable final Instant expiration, final boolean replace) throws ClientInformationManagerException {
        
        log.debug("Attempting to store client information (replace={})", replace ? "true" : "false");
        final String clientId = clientInformation.getID().getValue();
        //TODO: configurable serialization
        final String serialized = clientInformation.toJSONObject().toJSONString();
        try {
            if (getStorageService().create(CONTEXT_NAME, clientId, serialized,
                    expiration != null ? expiration.toEpochMilli() : null)) {
                log.info("Successfully stored the client information for ID {}", clientId);
            } else if (replace) {
                getStorageService().update(CONTEXT_NAME, clientId, serialized,
                        expiration != null ? expiration.toEpochMilli() : null);
            } else {
                throw new ClientInformationManagerException(
                        "Attempt to store duplicate client information for ID " + clientId);
            }
        } catch (final IOException e) {
            throw new ClientInformationManagerException("Could not store the client information", e);
        }
    }

    /** {@inheritDoc} */
    @Override
    @Deprecated
    public void storeClientInformation(@Nonnull final OIDCClientInformation clientInformation,
            @Nullable final Instant expiration) throws ClientInformationManagerException {
        storeClientInformation(clientInformation, expiration, false);
    }

    /** {@inheritDoc} */
    @Override
    public void destroyClientInformation(final ClientID clientId) {
        if (clientId == null) {
            log.warn("The null clientId cannot be destroyed, nothing to do");
            return;
        }
        try {
            getStorageService().delete(CONTEXT_NAME, clientId.getValue());
        } catch (final IOException e) {
            log.error("Could not delete the client ID {}", clientId.getValue(), e);
        }
    }

}