/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.util.Collections;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;

import net.shibboleth.oidc.metadata.ProviderMetadataResolver;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;
import net.shibboleth.utilities.java.support.service.ReloadableService;
import net.shibboleth.utilities.java.support.service.ServiceableComponent;


/**
 * This class uses the service interface to implement {@link ProviderMetadataResolver}.
 */
public class ReloadingProviderMetadataProvider extends AbstractIdentifiableInitializableComponent 
    implements ProviderMetadataResolver {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ReloadingProviderMetadataProvider.class);
    
    /** The service which managed the reloading. */
    private final ReloadableService<ProviderMetadataResolver> service;
    
    /**
     * Constructor.
     * 
     * @param resolverService the service which will manage the loading.
     */
    public ReloadingProviderMetadataProvider(
            @Nonnull final ReloadableService<ProviderMetadataResolver> resolverService) {
        service = Constraint.isNotNull(resolverService, "ProviderMetadataResolver Service cannot be null");
    }

    @Override
    public Iterable<OIDCProviderMetadata> resolve(@Nonnull final CriteriaSet criteria) throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ServiceableComponent<ProviderMetadataResolver> component = null;
        try {
            component = service.getServiceableComponent();
            if (null == component) {
                log.error("ReloadingProviderMetadataProvider '{}': Error accessing underlying source: "
                        + "Invalid configuration.", getId());
            } else {
                final ProviderMetadataResolver resolver = component.getComponent();
                return resolver.resolve(criteria);
            }
        } catch (final ResolverException e) {
            log.error("ProviderMetadataResolver '{}': Error during resolution", getId(), e);
        } finally {
            if (null != component) {
                component.unpinComponent();
            }
        }
        return Collections.emptySet();
    }

    @Override
    public OIDCProviderMetadata resolveSingle(@Nonnull final CriteriaSet criteria) throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);
        ServiceableComponent<ProviderMetadataResolver> component = null;
        try {
            component = service.getServiceableComponent();
            if (null == component) {
                log.error("ReloadingProviderMetadataProvider '{}': Error accessing underlying source: "
                        + "Invalid configuration.", getId());
            } else {
                final ProviderMetadataResolver resolver = component.getComponent();
                return resolver.resolveSingle(criteria);
            }
        } catch (final ResolverException e) {
            log.error("ProviderMetadataResolver '{}': Error during resolution", getId(), e);
        } finally {
            if (null != component) {
                component.unpinComponent();
            }
        }
        return null;
    }

}
