/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.io.IOException;
import java.util.Set;
import java.util.function.BiFunction;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.net.MediaType;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;

import net.shibboleth.oidc.metadata.criterion.IssuerIDCriterion;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.net.MediaTypeSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A dynamic strategy for fetching OpenID Connect Provider Configuration Metadata from its well-known location
 * (OpenID Connect Discovery 1.0, section 4). 
 */
@ThreadSafe
public class HTTPProviderConfigurationFetchingStrategy 
                            extends AbstractDynamicHTTPFetchingStrategy<OIDCProviderMetadata> {
    
    /** The returned content_type, must be application/json see openid-connect-discovery section 4.*/
    @Nonnull private static final MediaType CONTENT_TYPE = MediaType.JSON_UTF_8;
    
    /** The default well-known path for OpenID Provider metadata.  */
    @Nonnull @NotEmpty 
    private static final String DEFAULT_OPENID_PROVIDER_WELL_KNOWN_PATH = "/.well-known/openid-configuration";

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(HTTPProviderConfigurationFetchingStrategy.class);
 
    /** The well-known path for OpenID Provider metadata.  */
    @Nonnull @NotEmpty private String wellKnownPath;
    
    /** Strategy for composing an issuer with the well-known configuration path.*/
    @Nonnull private BiFunction<Issuer, String, String> wellKnownLocationCompositionStrategy;

    protected HTTPProviderConfigurationFetchingStrategy(@Nonnull final HttpClient client,
            @Nonnull final ResponseHandler<OIDCProviderMetadata> handler) {
        super(client, handler);
        wellKnownPath = DEFAULT_OPENID_PROVIDER_WELL_KNOWN_PATH;
        wellKnownLocationCompositionStrategy = new DefaultWellKnownPathCompositionStrategy();
    }
    
    /**
     * Set the well-known URL path component.
     * 
     * <p>Defaults to {@value DEFAULT_OPENID_PROVIDER_WELL_KNOWN_PATH}.</p>
     * 
     * @param path the path.
     */
    public void setWellKnownPath(@Nonnull @NotEmpty final String path) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        wellKnownPath = Constraint.isNotEmpty(path, "Well-know path can not be null");
    }
    
    /**
     * Set the well-known location composition strategy.
     * 
     * @param strategy the strategy to set.
     */
    public void setWellKnownLocationCompositionStrategy(@Nonnull final
            BiFunction<Issuer, String, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        wellKnownLocationCompositionStrategy = 
                Constraint.isNotNull(strategy, "Well known location composition strategy can not be null");
    }

    @Override
    protected String buildRequestURL(@Nonnull final CriteriaSet criteria) {
        if (criteria.contains(IssuerIDCriterion.class)) {
            final String url = wellKnownLocationCompositionStrategy
                    .apply(criteria.get(IssuerIDCriterion.class).getIssuerID(),wellKnownPath);
            
            log.debug("URL generated by request builder was: {}", url);            
            return url;
        }
        return null;
    }
    
    /** Default strategy for composing a well-known URL to fetch a provider's configuration document from.*/
    @Immutable
    @ThreadSafe
    private static final class DefaultWellKnownPathCompositionStrategy implements BiFunction<Issuer, String, String> {
        
        /** Class logger. */
        @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultWellKnownPathCompositionStrategy.class);

        @Override
        @Nullable public String apply(@Nonnull final Issuer issuer, @Nonnull @NotEmpty final String wellKnownPath) {
            // remove trailing slash if any (see openid-connect-discovery 4.1)
            final String normalizedIssuer = StringUtils.removeEnd(issuer.getValue(), "/");
            final StringBuilder builder = new StringBuilder();
            builder.append(normalizedIssuer).append(wellKnownPath);
            return builder.toString();
        }       
    }

    /** The response handler for parsing the providers's configuration information into {@link OIDCProviderMetadata}.*/
    @Immutable
    @ThreadSafe
    public static final class OIDCProviderMetadataResponseHandler 
                                                implements ResponseHandler<OIDCProviderMetadata> {        
        
        /** Class logger. */
        @Nonnull private final Logger log = LoggerFactory.getLogger(OIDCProviderMetadataResponseHandler.class);

        @Override
        @Nullable public OIDCProviderMetadata handleResponse(final HttpResponse response) throws IOException {
            
            final int httpStatusCode = response.getStatusLine().getStatusCode();
            
            final String currentRequestURI = MDC.get(MDC_ATTRIB_CURRENT_REQUEST_URI);
            
            // TODO we would need to do a conditional GET.
            if (httpStatusCode == HttpStatus.SC_NOT_MODIFIED) {
                log.debug("Metadata document from '{}' has not changed since last retrieval", 
                        currentRequestURI);
                return null;
            }
            
            if (httpStatusCode != HttpStatus.SC_OK) {
                log.warn("Non-ok status code '{}' returned from remote metadata source: {}",
                        httpStatusCode, currentRequestURI);
                return null;
            }
            
            try {
                validateHttpResponse(response);
            } catch (final ResolverException e) {
                log.error("Problem validating dynamic OIDC metadata HTTP response", e);
                return null;
            }
            
            try {              
                // this should convert the entity with the character set from the entity.
                final String jsonDocument = EntityUtils.toString(response.getEntity());                
                return OIDCProviderMetadata.parse(jsonDocument);
                
            } catch (final Exception e) {
                // catch any of the many exceptions
                log.error("Error parsing HTTP response stream", e);
                return null;
            }
            
        }
        
        /**
         * Validate the received HTTP response instance, such as checking for supported content types.
         * 
         * @param response the received response
         * @throws ResolverException if the response was not valid, or if there is a fatal error validating the response
         */
        protected void validateHttpResponse(@Nonnull final HttpResponse response) throws ResolverException {

            String contentTypeValue = null;
            final Header contentType = response.getEntity().getContentType();
            if (contentType != null && contentType.getValue() != null) {
                contentTypeValue = StringSupport.trimOrNull(contentType.getValue());
            }
            log.debug("Saw raw Content-Type from response header '{}'", contentTypeValue);
            
            if (!MediaTypeSupport.validateContentType(contentTypeValue,Set.of(CONTENT_TYPE), true, false)) {
                throw new ResolverException("HTTP response specified an unsupported Content-Type MIME type: " 
                        + contentTypeValue);
            }
            
        }
        
    }

}
