/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.saml.criterion.RoleDescriptorCriterion;
import org.opensaml.saml.metadata.resolver.RoleDescriptorResolver;
import org.opensaml.saml.metadata.resolver.filter.FilterException;
import org.opensaml.saml.metadata.resolver.filter.MetadataNodeProcessor;
import org.opensaml.saml.saml2.core.Audience;
import org.opensaml.saml.saml2.metadata.AssertionConsumerService;
import org.opensaml.saml.saml2.metadata.EntityDescriptor;
import org.opensaml.saml.saml2.metadata.Extensions;
import org.opensaml.saml.saml2.metadata.NameIDFormat;
import org.opensaml.saml.saml2.metadata.RoleDescriptor;
import org.opensaml.saml.saml2.metadata.SPSSODescriptor;
import org.opensaml.saml.security.impl.MetadataCredentialResolver;
import org.opensaml.security.credential.Credential;
import org.opensaml.xmlsec.keyinfo.KeyInfoCredentialResolver;
import org.opensaml.xmlsec.keyinfo.impl.BasicProviderKeyInfoCredentialResolver;
import org.opensaml.xmlsec.keyinfo.impl.KeyInfoProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jose.EncryptionMethod;
import com.nimbusds.jose.JWEAlgorithm;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.oauth2.sdk.GrantType;
import com.nimbusds.oauth2.sdk.ResponseType;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.SoftwareID;
import com.nimbusds.oauth2.sdk.id.SoftwareVersion;
import com.nimbusds.openid.connect.sdk.SubjectType;
import com.nimbusds.openid.connect.sdk.claims.ACR;
import com.nimbusds.openid.connect.sdk.rp.ApplicationType;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

import net.shibboleth.oidc.saml.xmlobject.Constants;
import net.shibboleth.oidc.saml.xmlobject.DefaultAcrValue;
import net.shibboleth.oidc.saml.xmlobject.MetadataValueSAMLObject;
import net.shibboleth.oidc.saml.xmlobject.OAuthRPExtensions;
import net.shibboleth.oidc.security.credential.JWKReferenceCredential;
import net.shibboleth.oidc.security.credential.NimbusSecretCredential;
import net.shibboleth.oidc.security.impl.CredentialConversionUtil;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * An implementation of {@link MetadataNodeProcessor} which supports adding an instance of
 * {@link OIDCClientInformation} to the object metadata of {@link SPSSODescriptor}. The data
 * is fetched mainly via {@link OAuthRPExtensions}.
 */
public class ClientInformationNodeProcessor implements MetadataNodeProcessor {

    /** The ACS binding identifier matching to the redirect_uri. */
    public static final String BINDING_ID_REDIRECT_URI = "https://tools.ietf.org/html/rfc6749#section-3.1.2";
    
    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(ClientInformationNodeProcessor.class);
    
    /** The {@link KeyInfoCredentialResolver} to be used for the resolution. */
    private final @Nonnull KeyInfoCredentialResolver keyInfoCredentialResolver;
    
    /**
     * Constructor.
     * 
     * @param keyInfoProviders The list of key info providers.
     */
    public ClientInformationNodeProcessor(@Nonnull final List<KeyInfoProvider> keyInfoProviders) {
        keyInfoCredentialResolver = new BasicProviderKeyInfoCredentialResolver(keyInfoProviders);
    }
    
    /** {@inheritDoc} */
    @Override
    public void process(final @Nullable XMLObject metadataNode) throws FilterException {
        if (metadataNode instanceof SPSSODescriptor) {
            final SPSSODescriptor roleDescriptor = (SPSSODescriptor) metadataNode;
            if (roleDescriptor.isSupportedProtocol(Constants.OIDC_PROTOCOL_URI)) {
                final ClientID clientId = parseClientID(roleDescriptor);
                if (clientId == null || StringSupport.trimOrNull(clientId.getValue()) == null) {
                    log.error("Could not find a value for client_id, nothing to do");
                    return;
                }
                final Iterable<Credential> credentials = resolveCredentials(roleDescriptor);
                final OIDCClientMetadata metadata = populateMetadata(roleDescriptor, credentials, clientId.getValue());
                final Secret clientSecret = parseClientSecret(credentials);
                final OIDCClientInformation clientInformation =
                        new OIDCClientInformation(clientId, null, metadata, clientSecret);
                metadataNode.getObjectMetadata().put(clientInformation);
            }
        }
    }

    /**
     * Converts the entityID of the given {@link SPSSODescriptor} into a {@link ClientID}. The value is fetched from
     * the {@link EntityDescriptor}, expected to be the parent element of the given role decriptor.
     * 
     * @param roleDescriptor The {@link SPSSODescriptor} to be used as a source.
     * @return The entityID value as {@link ClientID}.
     */
    protected @Nullable ClientID parseClientID(final @Nonnull SPSSODescriptor roleDescriptor) {
        if (!roleDescriptor.hasParent() || !(roleDescriptor.getParent() instanceof EntityDescriptor)) {
            log.warn("Unexpected structure, EntityDescriptor not as a parent for OAuthRPRoleDescriptor");
            return null;
        }
        final EntityDescriptor entityDescriptor = (EntityDescriptor) roleDescriptor.getParent();
        return new ClientID(entityDescriptor.getEntityID());
    }
    
    /**
     * Fetches the client secret from given the set of {@link Credential}s. The first credential matching the type
     * {@link NimbusSecretCredential} is used as the source.
     * 
     * @param credentials The source set of {@link Credential}s.
     * @return The client secret as {@link Secret}.
     */
    protected @Nullable Secret parseClientSecret(final @Nonnull Iterable<Credential> credentials) {
        for (final Credential credential : credentials) {
            log.trace("Processing credential type {}", credential.getCredentialType());
            if (NimbusSecretCredential.class.isAssignableFrom(credential.getCredentialType())) {
                log.debug("Found client secret from the credentials");
                return ((NimbusSecretCredential) credential).getSecret();
            }
        }
        log.trace("No client secret found from the credentials");
        return null;
    }

    /**
     * Populates the {@link OIDCClientMetadata} using the values found from the given {@link SPSSODescriptor}, the set
     * of {@link Credential}s and the client ID.
     * 
     * @param roleDescriptor The {@link SPSSODescriptor} to be used as a source.
     * @param credentials The source set of {@link Credential}s to be used for client secret and remote/local JWKS.
     * @param clientId The client ID.
     * @return The {@link OIDCClientMetadata} parsed from the given parameters.
     */
    protected @Nonnull OIDCClientMetadata populateMetadata(final @Nonnull SPSSODescriptor roleDescriptor,
            final @Nonnull Iterable<Credential> credentials, final @Nonnull String clientId) {
        final OIDCClientMetadata metadata = new OIDCClientMetadata();
        final OAuthRPExtensions extensions = getOAuthRPExtensions(roleDescriptor);
        if (extensions != null) {
            metadata.setApplicationType(parseApplicationType(extensions));
            metadata.setURI(getSingleURIValue(extensions.getClientUri()));
            metadata.setDefaultACRs(parseDefaultAcrValues(extensions));
            metadata.setGrantTypes(parseGrantTypes(extensions));
            metadata.setIDTokenJWEAlg(parseJweAlgorithm(extensions.getIdTokenEncryptedResponseAlg()));
            metadata.setIDTokenJWEEnc(parseEncryptionMethod(extensions.getIdTokenEncryptedResponseEnc()));
            metadata.setIDTokenJWSAlg(parseJwsAlgorithm(extensions.getIdTokenSignedResponseAlg()));
            metadata.setInitiateLoginURI(getSingleURIValue(extensions.getInitiateLoginUri()));
            metadata.setPostLogoutRedirectionURIs(parseUris(extensions.getPostLogoutRedirectUris()));
            metadata.setRedirectionURIs(parseRedirectUris(roleDescriptor));
            metadata.setRequestObjectJWEAlg(parseJweAlgorithm(extensions.getRequestObjectEncryptionAlg()));
            metadata.setRequestObjectJWEEnc(parseEncryptionMethod(extensions.getRequestObjectEncryptionEnc()));
            metadata.setRequestObjectJWSAlg(parseJwsAlgorithm(extensions.getRequestObjectSigningAlg()));
            metadata.setRequestObjectURIs(parseUris(extensions.getRequestUris()));
            metadata.setResponseTypes(parseResponseTypes(extensions));
            metadata.setScope(parseScopes(extensions));
            metadata.setSectorIDURI(getSingleURIValue(extensions.getSectorIdentifierUri()));
            final String softwareId = extensions.getSoftwareId();
            if (softwareId != null) {
                metadata.setSoftwareID(new SoftwareID(softwareId));
            }
            final String softwareVersion = extensions.getSoftwareVersion();
            if (softwareVersion != null) {
                metadata.setSoftwareVersion(new SoftwareVersion(softwareVersion));
            }
            if (extensions.getDefaultMaxAge() > 0) {
                metadata.setDefaultMaxAge(extensions.getDefaultMaxAge());
            }
            metadata.requiresAuthTime(extensions.isRequireAuthTime());
            metadata.setSubjectType(parseSubjectType(roleDescriptor));
            metadata.setTokenEndpointAuthMethod(parseClientAuthenticationMethod(extensions));
            metadata.setTokenEndpointAuthJWSAlg(parseJwsAlgorithm(extensions.getTokenEndpointAuthSigningAlg()));
            metadata.setUserInfoJWEAlg(parseJweAlgorithm(extensions.getUserInfoEncryptedResponseAlg()));
            metadata.setUserInfoJWEEnc(parseEncryptionMethod(extensions.getUserInfoEncryptedResponseEnc()));
            metadata.setUserInfoJWSAlg(parseJwsAlgorithm(extensions.getUserInfoSignedResponseAlg()));
            metadata.setJWKSetURI(parseJwkUri(credentials, clientId));
            if (metadata.getJWKSetURI() == null) {
                metadata.setJWKSet(parseJwkSet(credentials, clientId));
            }
            metadata.setCustomField("audience", parseAudiences(extensions));
        } else {
            log.debug("No {} found to be processed", OAuthRPExtensions.TYPE_LOCAL_NAME);
        }
        return metadata;
    }
    
    /**
     * Get the {@link OAuthRPExtensions} from the given {@link SPSSODescriptor}, it it was found from its extensions.
     * 
     * @param roleDescriptor The role descriptor to get the extensions from.
     * @return The extensions, if they were found from the role descriptor. <code>null</code> otherwise.
     */
    protected @Nullable OAuthRPExtensions getOAuthRPExtensions(@Nonnull final SPSSODescriptor roleDescriptor) {
        final Extensions extensions = roleDescriptor.getExtensions();
        if (extensions == null) {
            log.debug("No extensions found from the given SPSSODescriptor");
            return null;
        }
        final List<XMLObject> rpExtensions = extensions.getUnknownXMLObjects(OAuthRPExtensions.TYPE_NAME);
        if (rpExtensions == null || rpExtensions.isEmpty()) {
            log.debug("SPSSODescriptor Extensions element had no {} child elements", OAuthRPExtensions.TYPE_LOCAL_NAME);
            return null;
        }
        if (rpExtensions.size() > 1) {
            log.warn("More than one {} defined, using only one of them", OAuthRPExtensions.TYPE_LOCAL_NAME);
        }
        if (rpExtensions.get(0) instanceof OAuthRPExtensions) {
            log.debug("Successfully parsed {}", OAuthRPExtensions.TYPE_LOCAL_NAME);
            return (OAuthRPExtensions) rpExtensions.get(0);
        }
        log.warn("Could not parse {} from the element", OAuthRPExtensions.TYPE_LOCAL_NAME);
        return null;
    }
    
    /**
     * Get all the credentials attached to the given {@link SPSSODescriptor}. They are resolved using the
     * {@link #keyInfoCredentialResolver}.
     * 
     * @param roleDescriptor The role descriptor to parse the credentials from.
     * @return All the resolved credentials. Or empty set if none was found.
     */
    protected @Nonnull Iterable<Credential> resolveCredentials(@Nonnull final SPSSODescriptor roleDescriptor) {
        final MetadataCredentialResolver credentialResolver = new MetadataCredentialResolver();
        credentialResolver.setKeyInfoCredentialResolver(keyInfoCredentialResolver);
        credentialResolver.setRoleDescriptorResolver(new SkeletonEchoingRoleDescriptorResolver() {

            /** {@inheritDoc} */
            @Override public RoleDescriptor resolveSingle(final CriteriaSet criteria) throws ResolverException {
                return roleDescriptor;
            }

        });

        final RoleDescriptorCriterion criterion = new RoleDescriptorCriterion(roleDescriptor);
        final CriteriaSet criteriaSet = new CriteriaSet();
        criteriaSet.add(criterion);
        try {
            credentialResolver.initialize();
        } catch (final ComponentInitializationException e) {
            log.error("Could not initialize the SAML metadata credential resolver, cannot resolve JWKSet", e);
        }
        try {
            return credentialResolver.resolve(criteriaSet);
        } catch (final ResolverException e) {
            log.warn("Could not resolve credentials", e);
        }
        return Collections.emptySet();
    }
    
    /**
     * Convert the given credentials into the Nimbus {@link JWKSet}.
     * 
     * @param credentials The set to be converted.
     * @param clientId The client ID related to the credentials.
     * @return The given credentials converted into a JWKSet.
     */
    @Nullable protected JWKSet parseJwkSet(@Nonnull final Iterable<Credential> credentials, 
            final @Nonnull String clientId) {
        final List<JWK> jwks = new ArrayList<>();
        for (final Credential credential : credentials) {
            final JWK jwk = CredentialConversionUtil.credentialToKey(credential);
            if (jwk == null) {
                log.debug("Could not parse credential of {} to a JWK", clientId);
            } else {
                log.trace("Successfully parsed a JWK to client {}: {}", clientId, jwk.toJSONString());
                jwks.add(jwk);
            }
        }
        return jwks.isEmpty() ? null : new JWKSet(jwks);
    }

    /**
     * Convert the given credentials into a JWKS URI.
     * 
     * @param credentials The set to be converted.
     * @param clientId The client ID related to the credentials.
     * @return The given credentials converted into a JWKS URI
     */
    @Nullable protected URI parseJwkUri(@Nonnull final Iterable<Credential> credentials, 
            final @Nonnull String clientId) {
        for (final Credential credential : credentials) {
            if (credential instanceof JWKReferenceCredential) {
                log.trace("Successfully located a JWKS URI for client {}: {}", clientId,
                        ((JWKReferenceCredential) credential).getReferenceURI());
                return ((JWKReferenceCredential) credential).getReferenceURI();
            }
        }
        return null;
    }

    /**
     * Parse the {@link ClientAuthenticationMethod} from the given extensions.
     * 
     * @param extensions The extensions to parse from.
     * @return The client authentication method, or <code>null</code> it was not found.
     */
    @Nullable protected ClientAuthenticationMethod parseClientAuthenticationMethod(
            final @Nonnull OAuthRPExtensions extensions) {
        final String metadataValue = StringSupport.trimOrNull(extensions.getTokenEndpointAuthMethod());
        if (metadataValue == null) {
            return null;
        }
        return ClientAuthenticationMethod.parse(metadataValue);
    }
    
    /**
     * Parse the {@link ApplicationType} from the given extensions.
     * 
     * @param extensions The extensions to parse from.
     * @return {@link ApplicationType#NATIVE} if it was defined in the extensions, {@link ApplicationType#WEB}
     * otherwise.
     */
    @Nonnull protected ApplicationType parseApplicationType(@Nonnull final OAuthRPExtensions extensions) {
        final String metadataValue = StringSupport.trimOrNull(extensions.getApplicationType());
        if (ApplicationType.NATIVE.toString().equalsIgnoreCase(metadataValue)) {
            return ApplicationType.NATIVE;
        }
        return ApplicationType.WEB;
    }
    
    /**
     * Parse the {@link SubjectType} from the given role descriptor's name ID formats.
     * 
     * @param roleDescriptor The role descriptor to parse from. Only the first nameID definition is taken into
     * consideration.
     * @return {@link SubjectType#PAIRWISE} if <code>pairwise</code> was defined as the name ID format.
     * {@link SubjectType#PUBLIC} otherwise.
     */
    @Nonnull protected SubjectType parseSubjectType(@Nonnull final SPSSODescriptor roleDescriptor) {
        final List<NameIDFormat> nameIdFormats = roleDescriptor.getNameIDFormats();
        if (nameIdFormats == null || nameIdFormats.isEmpty()) {
            log.warn("No NameIDFormat defined, using 'public'");
            return SubjectType.PUBLIC;
        }
        if (nameIdFormats.size() > 1) {
            log.warn("Multiple NameIDFormats defined, using first one with a known value");
        }
        
        for (final NameIDFormat format : nameIdFormats) {
            if (format != null) {
                if (Constants.OIDC_SUB_NAMEID_FORMAT_PUBLIC.equals(format.getURI())) {
                    return SubjectType.PUBLIC;
                } else if (Constants.OIDC_SUB_NAMEID_FORMAT_PAIRWISE.equals(format.getURI())) {
                    return SubjectType.PAIRWISE;
                }
            }
        }
        
        log.warn("No known NameIDFormats defined, using 'public'");
        return SubjectType.PUBLIC;
    }
    
    /**
     * Parse the default {@link ACR} values from the given extensions.
     *
     * @param extensions The extensions to parse from.
     * @return The list of ACR values that were found.
     */
    @Nonnull protected List<ACR> parseDefaultAcrValues(@Nonnull final OAuthRPExtensions extensions) {
        final List<ACR> acrs = new ArrayList<>();
        for (final DefaultAcrValue acr : extensions.getDefaultAcrValues()) {
            final String value = StringSupport.trimOrNull(acr.getValue());
            if (value != null) {
                acrs.add(new ACR(value));
            }
        }
        return acrs;
    }
    
    /**
     * Parse the {@link GrantType}s from the given extensions.
     * 
     * @param extensions The extensions to parse from.
     * @return The set of grant types that were found.
     */
    @Nonnull protected Set<GrantType> parseGrantTypes(@Nonnull final OAuthRPExtensions extensions) {
        final Set<GrantType> grantTypes = new HashSet<>();
        final Collection<String> values = getListValues(extensions.getGrantTypes());
        for (final String value : values) {
            grantTypes.add(new GrantType(value));
        }
        return grantTypes;
    }
    
    /**
     * Parse the {@link ResponseType}s from the given extensions.
     * 
     * @param extensions The extensions to parse from.
     * @return The set of response types that were found.
     */
    @Nonnull protected Set<ResponseType> parseResponseTypes(@Nonnull final OAuthRPExtensions extensions) {
        final Set<ResponseType> responseTypes = new HashSet<>();
        final Collection<String> values = getListValues(extensions.getResponseTypes());
        for (final String value : values) {
            responseTypes.add(new ResponseType(value.split("\\+")));
        }
        return responseTypes;
    }
    
    /**
     * Parse the {@link Scope} from the given extensions.
     * 
     * @param extensions The extensions to parse from.
     * @return The scope that was found.
     */
    @Nonnull protected Scope parseScopes(@Nonnull final OAuthRPExtensions extensions) {
        final Scope scope = new Scope();
        final Collection<String> values = getListValues(extensions.getScopes());
        for (final String value : values) {
            scope.add(value);
        }
        return scope;
    }
    
    /**
     * Parse the {@link JWEAlgorithm} from the given metadata value.
     * 
     * @param value The metadata value to parse from.
     * @return The JWE algorithm, or <code>null</code> if no value was found.
     */
    @Nullable protected JWEAlgorithm parseJweAlgorithm(@Nullable final String value) {
        if (value != null) {
            return new JWEAlgorithm(value);
        }
        return null;
    }

    /**
     * Parse the {@link JWSAlgorithm} from the given metadata value.
     * 
     * @param value The metadata value to parse from.
     * @return The JWS algorithm, or <code>null</code> if no value was found.
     */
    @Nullable protected JWSAlgorithm parseJwsAlgorithm(@Nullable final String value) {
        if (value != null) {
            return new JWSAlgorithm(value);
        }
        return null;
    }

    /**
     * Parse the {@link EncryptionMethod} from the given metadata value.
     * 
     * @param value The metadata value to parse from.
     * @return The encryption method, or <code>null</code> if no value was found.
     */
    @Nullable protected EncryptionMethod parseEncryptionMethod(@Nullable final String value) {
        if (value != null) {
            return new EncryptionMethod(value);
        }
        return null;
    }
    
    /**
     * Parse the redirection URIs from the given role descriptor. Only the assertion consumer service URLs whose
     * binding matches to {@link #BINDING_ID_REDIRECT_URI} are taken into consideration.
     * 
     * @param roleDescriptor The role descriptor to parse from.
     * @return The set of redirection URIs that were successfully parsed.
     */
    @Nonnull protected Set<URI> parseRedirectUris(@Nonnull final SPSSODescriptor roleDescriptor) {
        final Set<URI> uris = new HashSet<>();
        for (final AssertionConsumerService acs : roleDescriptor.getAssertionConsumerServices()) {
            if (BINDING_ID_REDIRECT_URI.equals(acs.getBinding())) {
                final URI uri = getSingleURIValue(acs.getLocation());
                if (uri != null) {
                    uris.add(uri);
                }
            }
        }
        return uris;
    }
    
    /**
     * Parse the URIs from the given list of metadata values.
     * 
     * @param listOfValues The list to parse from.
     * @return Set of URIs that were successfully parsed from the list.
     */
    @Nonnull protected Set<URI> parseUris(final @Nonnull List<? extends MetadataValueSAMLObject> listOfValues) {
        final Set<URI> uris = new HashSet<>();
        for (final MetadataValueSAMLObject value : listOfValues) {
            final URI uri = getSingleURIValue(value);
            if (uri != null) {
                uris.add(uri);
            }
        }
        return uris;
    }
    
    /**
     * Parse the SAML Audience elements.
     * 
     * @param extensions extension container
     * 
     * @return audience collection or null
     */
    @Nullable @NonnullElements protected List<String> parseAudiences(@Nonnull final OAuthRPExtensions extensions) {
        final List<String> auds = new ArrayList<>();
        for (final XMLObject aud : extensions.getUnknownXMLObjects(Audience.DEFAULT_ELEMENT_NAME)) {
            if (aud instanceof Audience) {
                if (((Audience) aud).getURI() != null) {
                    auds.add(((Audience) aud).getURI());
                }
            }
        }
        return auds.isEmpty() ? null : auds;
    }
    
    /**
     * Parse an XML value list from a metadata value object into a collection of strings.
     * 
     * @param metadataValue input object
     * 
     * @return possibly empty value collection
     */
    @Nonnull @NonnullElements
    protected Collection<String> getListValues(@Nullable final String metadataValue) {
        if (metadataValue != null) {
            return StringSupport.stringToList(metadataValue, " \t\n\r");
        }
        return Collections.emptyList();
    }
    
    /**
     * Converts the metadata value object value into a {@link URI}.
     * 
     * @param metadataValue The metadata object value to convert from.
     * @return The value as URI if it was successfully parsed, <code>null</code> otherwise.
     */
    @Nullable protected URI getSingleURIValue(@Nonnull final MetadataValueSAMLObject metadataValue) {
        return getSingleURIValue(metadataValue.getValue());
    }
    
    /**
     * Converts the given {@link String} into a {@link URI}.
     * 
     * @param value The raw string value.
     * @return The value as URI if it was successfully parsed, <code>null</code> otherwise.
     */
    @Nullable protected URI getSingleURIValue(@Nullable final String value) {
        if (value != null) {
            try {
                return new URI(value);
            } catch (final URISyntaxException e) {
                log.warn("Could not parse {} into an URI", value, e);
            }
        }
        return null;
    }
    
    protected abstract class SkeletonEchoingRoleDescriptorResolver implements RoleDescriptorResolver {

        /** {@inheritDoc} */
        @Override  public Iterable<RoleDescriptor> resolve(final CriteriaSet criteria) throws ResolverException {
            return Arrays.asList(resolveSingle(criteria));
        }

        /** {@inheritDoc} */
        @Override  public String getId() {
            return "EmbeddedLocalRoleDescriptorResolver";
        }

        /** {@inheritDoc} */
        @Override  public boolean isRequireValidMetadata() {
            return false;
        }

        /** {@inheritDoc} */
        @Override public void setRequireValidMetadata(final boolean requireValidMetadata) {
            // no op
        }         
    }
    
}