/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.op.OIDCProviderMetadata;

import net.shibboleth.oidc.metadata.ProviderMetadataResolver;
import net.shibboleth.oidc.metadata.RefreshableProviderMetadataResolver;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A provider metadata provider that uses registered resolvers, in turn, to answer queries.
 * 
 * The Iterable of provider metadata returned is the first non-null and non-empty Iterable found while iterating over
 * the registered resolvers in resolver list order.
 */
public class ChainingProviderMetadataResolver extends AbstractIdentifiableInitializableComponent implements 
    RefreshableProviderMetadataResolver {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ChainingProviderMetadataResolver.class);
    
    /** Registered resolvers. */
    @Nonnull @NonnullElements private List<ProviderMetadataResolver> resolvers;
    
    /** Constructor. */
    public ChainingProviderMetadataResolver() {
        resolvers = Collections.emptyList();
    }
    
    /**
     * Get an immutable the list of currently registered resolvers.
     * 
     * @return list of currently registered resolvers
     */
    @Nonnull @NonnullElements @Unmodifiable @NotLive public List<ProviderMetadataResolver> getResolvers() {
        return resolvers;
    }
    
    /**
     * Set the registered provider metadata resolvers.
     * 
     * @param newResolvers the provider metadata resolvers to use
     * 
     * @throws ResolverException thrown if there is a problem adding the provider metadata resolvers
     */
    public void setResolvers(@Nullable @NonnullElements final List<? extends ProviderMetadataResolver> newResolvers)
            throws ResolverException {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        if (newResolvers == null || newResolvers.isEmpty()) {
            resolvers = Collections.emptyList();
        } else {
            resolvers = List.copyOf(newResolvers);
        }
    }
    
    /** {@inheritDoc} */
    @Override
    @Nullable public OIDCProviderMetadata resolveSingle(@Nullable final CriteriaSet criteria) 
            throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);

        final Iterable<OIDCProviderMetadata> iterable = resolve(criteria);
        if (iterable != null) {
            final Iterator<OIDCProviderMetadata> iterator = iterable.iterator();
            if (iterator != null && iterator.hasNext()) {
                return iterator.next();
            }
        }
        return null;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Iterable<OIDCProviderMetadata> resolve(@Nullable final CriteriaSet criteria) 
            throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);

        for (final ProviderMetadataResolver resolver : resolvers) {
            try {
                final Iterable<OIDCProviderMetadata> providerMetadatas = resolver.resolve(criteria);
                if (providerMetadatas != null && providerMetadatas.iterator().hasNext()) {
                    return providerMetadatas;
                }
            } catch (final ResolverException e) {
                log.warn("Error retrieving provider metadata from resolver of type {}, proceeding to next resolver",
                        resolver.getClass().getName(), e);
                continue;
            }
        }

        return Collections.emptyList();
    }

    /** {@inheritDoc} */
    @Override public void refresh() throws ResolverException {
        for (final ProviderMetadataResolver resolver : resolvers) {
            if (resolver instanceof RefreshableProviderMetadataResolver) {
                ((RefreshableProviderMetadataResolver) resolver).refresh();
            }
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Instant getLastUpdate() {
        Instant ret = null;
        for (final ProviderMetadataResolver resolver : resolvers) {
            if (resolver instanceof RefreshableProviderMetadataResolver) {
                final Instant lastUpdate = ((RefreshableProviderMetadataResolver) resolver).getLastUpdate();
                if (ret == null || ret.isBefore(lastUpdate)) {
                    ret = lastUpdate;
                }
            }
        }
        
        return ret;
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Instant getLastRefresh() {
        Instant ret = null;
        for (final ProviderMetadataResolver resolver : resolvers) {
            if (resolver instanceof RefreshableProviderMetadataResolver) {
                final Instant lastRefresh = ((RefreshableProviderMetadataResolver) resolver).getLastRefresh();
                if (ret == null || ret.isBefore(lastRefresh)) {
                    ret = lastRefresh;
                }
            }
        }
        
        return ret;
    }
    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        if (resolvers == null) {
            log.warn("ProviderMetadataResolver was not configured with any member " + 
                    "ProviderMetadataResolvers");
            resolvers = Collections.emptyList();
        } else {
            final List<String> resolverDetails = new ArrayList<>();
            for (final ProviderMetadataResolver resolver : resolvers) {
                resolverDetails.add(resolver.getId() + ": " + countClients(resolver) + " clients");
            }
            log.info("ChainingProviderMetadataResolver was configured with the following resolvers: {}",
                    resolverDetails);
        }
    }

    /** {@inheritDoc} */
    @Override protected void doDestroy() {
        super.doDestroy();
        resolvers = Collections.emptyList();
    }
    
    /**
     * Counts the clients found from the given resolver.
     * 
     * @param resolver The resolver whose clients are counted.
     * @return The amount of resolvable clients.
     */
    protected int countClients(final ProviderMetadataResolver resolver) {
        int count = 0;
        final Iterable<OIDCProviderMetadata> iterable;
        try {
            iterable = resolver.resolve(new CriteriaSet());
        } catch (final ResolverException e) {
            log.warn("ChainingProviderMetadataResolver could not count clients for {}", resolver.getId());
            return 0;
        }
      
        final Iterator<OIDCProviderMetadata> iterator = iterable.iterator();
        while (iterator.hasNext()) {
            iterator.next();
            count++;
        }       
        return count;
    }

}
