/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

import net.shibboleth.oidc.metadata.ClientInformationResolver;
import net.shibboleth.oidc.metadata.RefreshableClientInformationResolver;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A client information provider that uses registered resolvers, in turn, to answer queries.
 * 
 * The Iterable of client informations returned is the first non-null and non-empty Iterable found while iterating over
 * the registered resolvers in resolver list order.
 */
public class ChainingClientInformationResolver extends AbstractIdentifiableInitializableComponent implements 
    RefreshableClientInformationResolver {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ChainingClientInformationResolver.class);
    
    /** Registered resolvers. */
    @Nonnull @NonnullElements private List<ClientInformationResolver> resolvers;

    /** Constructor. */
    public ChainingClientInformationResolver() {
        resolvers = Collections.emptyList();
    }
    
    /**
     * Get an immutable the list of currently registered resolvers.
     * 
     * @return list of currently registered resolvers
     */
    @Nonnull @NonnullElements @Unmodifiable @NotLive public List<ClientInformationResolver> getResolvers() {
        return resolvers;
    }

    /**
     * Set the registered client information resolvers.
     * 
     * @param newResolvers the client information resolvers to use
     * 
     * @throws ResolverException thrown if there is a problem adding the client information resolvers
     */
    public void setResolvers(@Nullable @NonnullElements final List<? extends ClientInformationResolver> newResolvers)
            throws ResolverException {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        if (newResolvers == null || newResolvers.isEmpty()) {
            resolvers = Collections.emptyList();
        } else {
            resolvers = List.copyOf(newResolvers);
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public OIDCClientInformation resolveSingle(@Nullable final CriteriaSet criteria) 
            throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);

        final Iterable<OIDCClientInformation> iterable = resolve(criteria);
        if (iterable != null) {
            final Iterator<OIDCClientInformation> iterator = iterable.iterator();
            if (iterator != null && iterator.hasNext()) {
                return iterator.next();
            }
        }
        return null;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Iterable<OIDCClientInformation> resolve(@Nullable final CriteriaSet criteria) 
            throws ResolverException {
        ComponentSupport.ifNotInitializedThrowUninitializedComponentException(this);

        for (final ClientInformationResolver resolver : resolvers) {
            try {
                final Iterable<OIDCClientInformation> clientInformations = resolver.resolve(criteria);
                if (clientInformations != null && clientInformations.iterator().hasNext()) {
                    return clientInformations;
                }
            } catch (final ResolverException e) {
                log.warn("Error retrieving client information from resolver of type {}, proceeding to next resolver",
                        resolver.getClass().getName(), e);
                continue;
            }
        }

        return Collections.emptyList();
    }

    /** {@inheritDoc} */
    @Override public void refresh() throws ResolverException {
        for (final ClientInformationResolver resolver : resolvers) {
            if (resolver instanceof RefreshableClientInformationResolver) {
                ((RefreshableClientInformationResolver) resolver).refresh();
            }
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Instant getLastUpdate() {
        Instant ret = null;
        for (final ClientInformationResolver resolver : resolvers) {
            if (resolver instanceof RefreshableClientInformationResolver) {
                final Instant lastUpdate = ((RefreshableClientInformationResolver) resolver).getLastUpdate();
                if (ret == null || ret.isBefore(lastUpdate)) {
                    ret = lastUpdate;
                }
            }
        }
        
        return ret;
    }

    /** {@inheritDoc} */
    @Override
    @Nullable public Instant getLastRefresh() {
        Instant ret = null;
        for (final ClientInformationResolver resolver : resolvers) {
            if (resolver instanceof RefreshableClientInformationResolver) {
                final Instant lastRefresh = ((RefreshableClientInformationResolver) resolver).getLastRefresh();
                if (ret == null || ret.isBefore(lastRefresh)) {
                    ret = lastRefresh;
                }
            }
        }
        
        return ret;
    }
    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        if (resolvers == null) {
            log.warn("ChainingClientInformationResolver was not configured with any member " + 
                    "ClientInformationResolvers");
            resolvers = Collections.emptyList();
        } else {
            final List<String> resolverDetails = new ArrayList<>();
            for (final ClientInformationResolver resolver : resolvers) {
                resolverDetails.add(resolver.getId() + ": " + countClients(resolver) + " clients");
            }
            log.info("ChainingClientInformationResolver was configured with the following resolvers: {}",
                    resolverDetails);
        }
    }

    /** {@inheritDoc} */
    @Override protected void doDestroy() {
        super.doDestroy();
        resolvers = Collections.emptyList();
    }
    
    /**
     * Counts the clients found from the given resolver.
     * 
     * @param resolver The resolver whose clients are counted.
     * @return The amount of resolvable clients.
     */
    protected int countClients(final ClientInformationResolver resolver) {
        int count = 0;
        final Iterable<OIDCClientInformation> iterable;
        try {
            iterable = resolver.resolve(new CriteriaSet());
        } catch (final ResolverException e) {
            log.warn("ChainingClientInformationResolver could not count clients for {}", resolver.getId());
            return 0;
        }
        if (iterable != null) {
            final Iterator<OIDCClientInformation> iterator = iterable.iterator();
            while (iterator.hasNext()) {
                iterator.next();
                count++;
            }
        }
        return count;
    }

}