/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;
import com.nimbusds.oauth2.sdk.id.Identifier;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.Resolver;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * A base class for {@link Resolver}s used for resolving entities containing identifiers based on {@link Identifier}. 
 *
 * @param <Key> The identifier type in the backing store
 * @param <Value> The entity type in the backing store
 */
public abstract class AbstractOIDCEntityResolver<Key extends Identifier, Value> 
    extends AbstractIdentifiableInitializableComponent {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(AbstractOIDCEntityResolver.class);
    
    /** Backing store for runtime JSON data. */
    private JsonBackingStore jsonBackingStore;
    
    /** Logging prefix. */
    private String logPrefix;
    
    /**
     * Whether problems during initialization should cause the provider to fail or go on without metadata. The
     * assumption being that in most cases a provider will recover at some point in the future. Default: true.
     */
    private boolean failFastInitialization;
    
    /**
     * Constructor.
     */
    public AbstractOIDCEntityResolver() {
        failFastInitialization = true;
    }
    
    /**
     * Return a prefix for logging messages for this component.
     * 
     * @return a string for insertion at the beginning of any log messages
     */
    @Nonnull @NotEmpty protected String getLogPrefix() {
        if (logPrefix == null) {
            logPrefix = String.format("Metadata Resolver %s %s:", getClass().getSimpleName(), getId());
        }
        return logPrefix;
    }

    /**
     * Gets whether problems during initialization should cause the provider to fail or go on without metadata. The
     * assumption being that in most cases a provider will recover at some point in the future.
     * 
     * @return whether problems during initialization should cause the provider to fail
     */
    public boolean isFailFastInitialization() {
        return failFastInitialization;
    }

    /**
     * Sets whether problems during initialization should cause the provider to fail or go on without metadata. The
     * assumption being that in most cases a provider will recover at some point in the future.
     * 
     * @param failFast whether problems during initialization should cause the provider to fail
     */
    public void setFailFastInitialization(final boolean failFast) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        failFastInitialization = failFast;
    }

    
    /** {@inheritDoc} */
    @Override protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        try {
            initOIDCResolver();
        } catch (final ComponentInitializationException e) {
            if (failFastInitialization) {
                log.error("OIDC metadata provider failed to properly initialize, fail-fast=true, halting", e);
                throw e;
            } else {
                log.error("OIDC metadata provider failed to properly initialize, fail-fast=false, "
                        + "continuing on in a degraded state", e);
            }
        }
    }
    
    /**
     * Initializes this resolver by creating a new backing store.
     * @throws ComponentInitializationException If the initialization fails.
     */
    protected void initOIDCResolver() throws ComponentInitializationException {
        jsonBackingStore = createNewBackingStore();
    }

    /**
     * Get list of information matching a given identifier.
     * 
     * @param identifier identifier to lookup
     * @return a list of information
     * @throws ResolverException if an error occurs
     */
    @Nonnull @NonnullElements protected List<Value> lookupIdentifier(
            @Nonnull @NotEmpty final Key identifier)
            throws ResolverException {
        if (!isInitialized()) {
            throw new ResolverException("Metadata resolver has not been initialized");
        }

        if (identifier == null || Strings.isNullOrEmpty(identifier.getValue())) {
            log.debug("Identifier was null or empty, skipping search for it");
            return Collections.emptyList();
        }

        final List<Value> allInformation = lookupIndexedIdentifier(identifier);
        if (allInformation.isEmpty()) {
            log.debug("Backing store does not contain any information with the ID: {}", identifier);
            return allInformation;
        }
        return allInformation;
    }

    /**
     * Lookup the specified identifier from the index. The returned list will be a copy of what is stored in the 
     * backing index, and is safe to be manipulated by callers.
     * 
     * @param identifier the identifier to lookup
     * 
     * @return list copy of indexed identifiers, may be empty, will never be null
     */
    @Nonnull @NonnullElements protected List<Value> lookupIndexedIdentifier(
            @Nonnull @NotEmpty final Key identifier) {
        final List<Value> allInformation = getBackingStore().getIndexedInformation().get(identifier);
        if (allInformation != null) {
            return new ArrayList<>(allInformation);
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * Pre-process the specified entity descriptor, updating the specified entity backing store instance as necessary.
     * 
     * @param entityDescriptor the target entity descriptor to process
     * @param key key to entity
     * @param backingStore the backing store instance to update
     */
    protected void preProcessEntityDescriptor(@Nonnull final Value entityDescriptor, @Nonnull final Key key,
            @Nonnull final JsonBackingStore backingStore) {

        backingStore.getOrderedInformation().add(entityDescriptor);
        indexEntityDescriptor(entityDescriptor, key, backingStore);
    }
    
    /**
     * Remove from the backing store all metadata for the entity with the given identifier.
     * 
     * @param identifier the identifier of the metadata to remove
     * @param backingStore the backing store instance to update
     */
    protected void removeByIdentifier(@Nonnull final Key identifier, @Nonnull final JsonBackingStore backingStore) {
        final Map<Key, List<Value>> indexedDescriptors = backingStore.getIndexedInformation();
        final List<Value> descriptors = indexedDescriptors.get(identifier);
        if (descriptors != null) {
            backingStore.getOrderedInformation().removeAll(descriptors);
        }
        indexedDescriptors.remove(identifier);
    }

    /**
     * Index the specified entity descriptor, updating the specified entity backing store instance as necessary.
     * 
     * @param entityDescriptor the target entity descriptor to process
     * @param key key to entity
     * @param backingStore the backing store instance to update
     */
    protected void indexEntityDescriptor(@Nonnull final Value entityDescriptor, @Nonnull final Key key,
            @Nonnull final JsonBackingStore backingStore) {

        List<Value> entities = backingStore.getIndexedInformation().get(key);
        if (entities == null) {
            entities = new ArrayList<>();
            backingStore.getIndexedInformation().put(key, entities);
        } else if (!entities.isEmpty()) {
            log.warn("Detected duplicate object for key: {}", key);
        }
        entities.add(entityDescriptor);
    }

    
    /**
     * Create a new backing store instance for entity data. Subclasses may override to return a more
     * specialized subclass type. Note this method does not make the returned backing store the effective one in use.
     * The caller is responsible for calling {@link #setBackingStore(AbstractOIDCEntityResolver.JsonBackingStore)}
     * to make it the effective instance in use.
     * 
     * @return the new backing store instance
     */
    @Nonnull protected JsonBackingStore createNewBackingStore() {
        return new JsonBackingStore();
    }

    /**
     * Get the entity backing store currently in use by the metadata resolver.
     * 
     * @return the current effective entity backing store
     */
    @Nonnull protected JsonBackingStore getBackingStore() {
        return jsonBackingStore;
    }

    /**
     * Set the entity backing store currently in use by the metadata resolver.
     * 
     * @param newBackingStore the new entity backing store
     */
    protected void setBackingStore(@Nonnull final JsonBackingStore newBackingStore) {
        jsonBackingStore = Constraint.isNotNull(newBackingStore, "JsonBackingStore may not be null");
    }

    
    /**
     * The collection of data which provides the backing store for the processed metadata.
     */
    protected class JsonBackingStore {

        /** Index of identifiers to their entity information. */
        private Map<Key, List<Value>> indexedEntities;

        /** Ordered list of entity information. */
        private List<Value> orderedEntitiess;

        /** Constructor. */
        protected JsonBackingStore() {
            indexedEntities = new ConcurrentHashMap<>();
            orderedEntitiess = new ArrayList<>();
        }

        /**
         * Get the entity information index.
         * 
         * @return the entity information index.
         */
        @Nonnull public Map<Key, List<Value>> getIndexedInformation() {
            return indexedEntities;
        }

        /**
         * Get the ordered entity information.
         * 
         * @return the entity information.
         */
        @Nonnull public List<Value> getOrderedInformation() {
            return orderedEntitiess;
        }

    }
}
