/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.impl;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.http.client.HttpClient;
import org.apache.http.client.ResponseHandler;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.opensaml.security.httpclient.HttpClientSecurityParameters;
import org.opensaml.security.httpclient.HttpClientSecuritySupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.base.Strings;
import com.google.common.net.MediaType;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.annotation.constraint.Unmodifiable;
import net.shibboleth.utilities.java.support.collection.LazySet;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;
import net.shibboleth.utilities.java.support.resolver.ResolverException;

/**
 * Abstract strategy for fetching metadata dynamically over HTTP.
 *
 * @param <MetadataType> the metadata type.
 */
//TODO the MDC stuff.
public abstract class AbstractDynamicHTTPFetchingStrategy<MetadataType> 
        extends AbstractIdentifiableInitializableComponent implements Function<CriteriaSet, MetadataType> {
    
    
    /** MDC attribute representing the current request URI. Will be available during the execution of the 
     * configured {@link ResponseHandler}. */
    public static final String MDC_ATTRIB_CURRENT_REQUEST_URI = 
            AbstractDynamicHTTPFetchingStrategy.class.getName() + ".currentRequestURI";
    
    /** Default list of supported content MIME types. */
    private static final String[] DEFAULT_CONTENT_TYPES = new String[] {"application/json",
            "application/samlmetadata+xml", "application/xml", "text/xml"}; 
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AbstractDynamicHTTPFetchingStrategy.class);
    
    /** HTTP Client used to pull the configuration information. */
    @Nonnull private final HttpClient httpClient;
    
    /** List of supported MIME types for use in Accept request header and validation of 
     * response Content-Type header.*/
    @NonnullAfterInit private List<String> supportedContentTypes;
    
    /** Generated Accept request header value. */
    @NonnullAfterInit private String supportedContentTypesValue;
    
    /**Supported {@link MediaType} instances, constructed from the {@link #supportedContentTypes} list. */
    @NonnullAfterInit private Set<MediaType> supportedMediaTypes;
    
    /** HTTP client security parameters. */
    @Nullable private HttpClientSecurityParameters httpClientSecurityParameters;
    
    /** HttpClient ResponseHandler instance to use. */
    @Nonnull private final ResponseHandler<MetadataType> responseHandler;
    
    /**
     * 
     * Constructor.
     *
     * @param client the instance of {@link HttpClient} used to fetch remote OIDC metadata
     * @param handler the response handler used to convert the HTTP response to the metadata type.
     */
    protected AbstractDynamicHTTPFetchingStrategy(@Nonnull final HttpClient client, 
            @Nonnull final ResponseHandler<MetadataType> handler) {
        responseHandler = Constraint.isNotNull(handler, "Response handler can not be null");
        httpClient = Constraint.isNotNull(client, "HTTP Client can not be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (getSupportedContentTypes() == null) {
            setSupportedContentTypes(Arrays.asList(DEFAULT_CONTENT_TYPES));
        }
        
        if (! getSupportedContentTypes().isEmpty()) {
            supportedContentTypesValue = StringSupport.listToStringValue(getSupportedContentTypes(), ", ");
            supportedMediaTypes = new LazySet<>();
            for (final String contentType : getSupportedContentTypes()) {
                supportedMediaTypes.add(MediaType.parse(contentType));
            }
        } else {
            supportedMediaTypes = Collections.emptySet();
        }
        
        log.debug("{} Supported content types are: {}", getId(), getSupportedContentTypes());
    }
    
    /**
     * Set an instance of {@link HttpClientSecurityParameters} which provides various parameters to influence
     * the security behavior of the HttpClient instance.
     * 
     * <p>
     * For all TLS-related parameters, must be used in conjunction with an HttpClient instance 
     * which is configured with either:
     * </p>
     * <ul>
     * <li>
     * a {@link net.shibboleth.utilities.java.support.httpclient.TLSSocketFactory}
     * </li>
     * <li>
     * a {@link org.opensaml.security.httpclient.impl.SecurityEnhancedTLSSocketFactory} which wraps
     * an instance of {@link net.shibboleth.utilities.java.support.httpclient.TLSSocketFactory}, with
     * the latter likely configured in a "no trust" configuration.  This variant is required if either a
     * trust engine or a client TLS credential is to be used.
     * </li>
     * </ul>
     *
     * <p>
     * For convenience methods for building a 
     * {@link net.shibboleth.utilities.java.support.httpclient.TLSSocketFactory}, 
     * see {@link net.shibboleth.utilities.java.support.httpclient.HttpClientSupport}.
     * </p>
     *
     * <p>
     * If the appropriate TLS socket factory is not configured and a trust engine is specified,
     * then this will result in no TLS trust evaluation being performed and a 
     * {@link ResolverException} will ultimately be thrown.
     * </p>
     *
     * @param params the security parameters
     */
    public void setHttpClientSecurityParameters(@Nullable final HttpClientSecurityParameters params) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        httpClientSecurityParameters = params;
    }
    
    /**
     * Get the list of supported MIME types for use in Accept request header and validation of 
     * response Content-Type header.
     * 
     * @return the supported content types
     */
    @NonnullAfterInit @NotLive @Unmodifiable
    public List<String> getSupportedContentTypes() {
        return supportedContentTypes;
    }
    
    /**
     * Set the list of supported MIME types for use in Accept request header and validation of 
     * response Content-Type header. Values will be effectively lower-cased at runtime.
     * 
     * @param types the new supported content types to set
     */
    public void setSupportedContentTypes(@Nullable final List<String> types) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        if (types == null) {
            supportedContentTypes = Collections.emptyList();
        } else {
            supportedContentTypes = StringSupport.normalizeStringCollection(types)
                    .stream()
                    .filter(s -> s != null)
                    .map(String::toLowerCase)
                    .collect(Collectors.toUnmodifiableList());
        }
    }

    @Override
    @Nullable public MetadataType apply(@Nonnull final CriteriaSet criteria) {
        log.info("{} fetching metadata based on criteria: {}", getId(), criteria);
        final HttpUriRequest request = buildHttpRequest(criteria);
        if (request == null) {
            log.debug("Could not build request based on input criteria, unable to query");
            return null;
        }
        
        final HttpClientContext context = buildHttpClientContext(request);
        
        try {
            MDC.put(MDC_ATTRIB_CURRENT_REQUEST_URI, request.getURI().toString());
            final MetadataType result = httpClient.execute(request, responseHandler, context);
            HttpClientSecuritySupport.checkTLSCredentialEvaluated(context, request.getURI().getScheme());
            return result;
        } catch (final IOException e) {
            log.warn("Unable to fetch metadata from remote HTTP source",e);
            return null;
        } finally {
            MDC.remove(MDC_ATTRIB_CURRENT_REQUEST_URI);
        }
    }
    
    /**
    * Build the {@link HttpClientContext} instance which will be used to invoke the {@link HttpClient} request.
    * 
    * @param request the current HTTP request
    * 
    * @return a new instance of {@link HttpClientContext}
    */
   private HttpClientContext buildHttpClientContext(@Nonnull final HttpUriRequest request) {
       final HttpClientContext context = HttpClientContext.create();
       
       HttpClientSecuritySupport.marshalSecurityParameters(context, httpClientSecurityParameters, true);
       HttpClientSecuritySupport.addDefaultTLSTrustEngineCriteria(context, request);
       
       return context;
   }
    
    /**
    * Build an appropriate instance of {@link HttpUriRequest} based on the input criteria set.
    * 
    * @param criteria the input criteria set
    * @return the newly constructed request, or null if it can not be built from the supplied criteria
    */
   @Nullable private HttpUriRequest buildHttpRequest(@Nonnull final CriteriaSet criteria) {
       final String url = buildRequestURL(criteria);
       log.debug("Built request URL of: {}", url);
       
       if (url == null) {
           log.debug("Could not construct request URL from input criteria, unable to query");
           return null;
       }
           
       final HttpGet getMethod = new HttpGet(url);
     
       if (!Strings.isNullOrEmpty(supportedContentTypesValue)) {
           getMethod.addHeader("Accept", supportedContentTypesValue);
       }
       
       return getMethod;
   }
    
    /**
     * Build the request URL based on the input criteria set.
     * 
     * @param criteria the input criteria set
     * @return the request URL, or null if it can not be built based on the supplied criteria
     */
    @Nullable protected abstract String buildRequestURL(@Nonnull final CriteriaSet criteria);

}
