/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.metadata.cache.MetadataCache;
import net.shibboleth.oidc.metadata.cache.MetadataCacheException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.NotLive;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A {@link MetadataCache metadata cache} implementation that does not support reading and writing to
 * a backing store. Entries are fetched, validated, and filtered before being returned to the caller.
 * Each request always requires fetching and processing. This is not,
 * therefore, a true cache, and is only useful for special cases which just want to exercise the
 * fetch, validate, filter, cycle. 
 *
 * @param <IdentifierType> the metadata identifier type.
 * @param <MetadataType> the metadata type.
 */
@ThreadSafe
public class FetchThroughMetadataCache <IdentifierType, MetadataType> 
                            extends AbstractMetadataCache<IdentifierType, MetadataType> {    

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(FetchThroughMetadataCache.class);
    
    /** The function to use to fetch/load metadata entries.*/
    @NonnullAfterInit private Function<CriteriaSet, MetadataType> fetchStrategy;

    /** Constructor. */
    protected FetchThroughMetadataCache() {
        // Does not require a backingstore.
        super(null);
    }
    
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();  

        if (fetchStrategy == null) {
            throw new ComponentInitializationException("Metadata fetching strategy can not be null");
        }
        
    }
    
    /**
     * Set the metadata fetching strategy. 
     * 
     * @param strategy the strategy used to fetch metadata using a 'read-through' semantic.
     */
    public void setFetchStrategy(@Nonnull final Function<CriteriaSet, MetadataType> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        fetchStrategy = Constraint.isNotNull(strategy, "Dynamic Metadata fetch strategy can not be null");
    }

    @Override
    @Nonnull @NonnullElements @NotLive public List<MetadataType> get(
            @Nonnull @NotEmpty final CriteriaSet criteria) throws MetadataCacheException {
        
        if (!isInitialized()) {
            throw new MetadataCacheException("Metadata cache has not been initialized");
        }

        final IdentifierType identifier = getCriteriaToIdentifierStrategy().apply(criteria);
        log.debug("{} Resolved criteria to identifier: {}", getLogPrefix(), identifier);
        
        if (identifier != null) {         
            final MetadataType resolvedMetadata = fetchStrategy.apply(criteria);  
    
            if (resolvedMetadata != null && getMetadataValidPredicate().test(resolvedMetadata)) {
                
                final MetadataType filteredMetadata = 
                        getMetadataFilterStrategy().apply(resolvedMetadata, newFilterContext());
                
                if (filteredMetadata == null) {
                    log.warn("{} Filtered metadata is null, no further processing performed", getLogPrefix());
                    return Collections.emptyList();
                }
                
                final IdentifierType extractedIdentifier = getIdentifierExtractionStrategy().apply(filteredMetadata);
                
                if (extractedIdentifier == null) {
                    log.warn("{} Metadata identifier could not be extracted, no further processing performed", 
                            getLogPrefix());
                    return Collections.emptyList();
                }
                
                // equality method of the identifier is required to be implemented correctly.
                if (!Objects.equals(identifier, extractedIdentifier)) {
                    log.warn("{} New metadata's identifer '{}' does not match expected identifier '{}', "
                            + "will not process", getLogPrefix(), extractedIdentifier, identifier); 
                    return Collections.emptyList();
                }                   
                
                log.debug("{} Resolved metadata with identifier '{}'",getLogPrefix(), extractedIdentifier);
                
                return List.of(filteredMetadata);
            } else {
                log.trace("{} Metadata for '{}' could not be resolved or is not valid",getLogPrefix(), identifier);
            }
        } else {
            log.debug("Identifier not resolvable from criteria, can not fetch metadata");            
        }
        return Collections.emptyList();
 
    }
    
}
