/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.metrics.MetricsSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.RatioGauge;
import com.codahale.metrics.Timer;
import com.codahale.metrics.Timer.Context;

import net.shibboleth.oidc.metadata.DynamicBackingStore;
import net.shibboleth.oidc.metadata.MetadataManagementData;
import net.shibboleth.oidc.metadata.cache.ExpirationTimeContext;
import net.shibboleth.oidc.metadata.cache.MetadataCacheException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A metadata cache implementation that supports 'read-through' semantics. Does not support 'refresh-ahead' semantics
 * for loading about to expire values asynchronously ahead of time.
 * 
 * <p>Supports the following:</p>
 * <ul>
 * <li>Read-write locking on individual metadata entries. Including optimistic reads.</li>
 * <li>Synchronous 'read-through' metadata fetching for new metadata.</li>
 * <li>Synchronous 'read-through' metadata fetching for stale (past refresh point) metadata.</li>
 * <li>A background task to remove expired and idle metadata.</li>
 * </ul>
 * 
 * <p>This class should only be instantiated through the {@link MetadataCacheBuilder}.</p>
 *
 * @param <IdentifierType> the metadata identifier type.
 * @param <MetadataType> the metadata type.
 */
public class DynamicMetadataCache<IdentifierType, MetadataType> 
                            extends AbstractMetadataCache<IdentifierType, MetadataType> {
    
    /** Metric name for the timer for {@link #get(CriteriaSet)}. */
    public static final String METRIC_TIMER_GET = "timer.get";
    
    /** Metric name for the gauge of the number of live entityIDs. */
    public static final String METRIC_GAUGE_NUM_LIVE_INDEX_METADATA = "gauge.numLiveIndexedMetadata";
    
    /** Metric name for the timer for {@link #fetch(MetadataManagementData, Object, CriteriaSet)}. */
    public static final String METRIC_TIMER_FETCH_FROM_ORIGIN_SOURCE = "timer.fetchFromOriginSource";
    
    /** Metric name for the ratio gauge of fetches to resolve requests. */
    public static final String METRIC_RATIOGAUGE_FETCH_TO_GET = "ratioGauge.fetchToGet";
    
    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(DynamicMetadataCache.class);
    
    /** The interval at which the cleanup task should run. */
    @NonnullAfterInit private Duration cleanupTaskInterval;
    
    /** The initial cleanup task delay.*/
    @NonnullAfterInit private Duration initialCleanupTaskDelay;      
    
    /** The maximum idle time for which the resolver will keep data for a given entityID, 
     * before it is removed. */
    @NonnullAfterInit private Duration maxIdleEntityData;
    
    /** Flag indicating whether idle entity data should be removed. */
    private boolean removeIdleEntityData;
    
    /** Minimum cache duration. */
    @NonnullAfterInit private Duration minCacheDuration;
    
    /** Maximum cache duration. */
    @NonnullAfterInit private Duration maxCacheDuration;
    
    /** The function to use to fetch/load metadata if either none exists, or the existing is stale.*/
    @NonnullAfterInit private Function<CriteriaSet, MetadataType> fetchStrategy;
    
    /** Strategy used to compute an expiration time from a metadata instance. */
    @NonnullAfterInit private Function<ExpirationTimeContext<MetadataType>, Instant> metadataExpirationTimeStrategy;
    
    /** Mapping function to use when creating new metadata management data.*/
    @Nonnull private final Function<IdentifierType, MetadataManagementData<IdentifierType>> mgmtMappingFunction;

    /** Base name for Metrics instrumentation names. */
    @NonnullAfterInit private String metricsBaseName;
    
    /** Metrics Timer for {@link #get(CriteriaSet)}. */
    @Nullable private Timer timerGet;
    
    /** Metrics Timer for {@link #fetch(MetadataManagementData, Object, CriteriaSet)}. */
    @Nullable private Timer timerFetchFromSource;
    
    /** Metrics RatioGauge for count of origin fetches to gets.*/
    @Nullable private RatioGauge ratioGaugeFetchToGet;
    
    /** Metrics Gauge for the number of live indexed metadata.*/
    @Nullable private Gauge<Integer> gaugeNumLiveIndexedMetadata;

    /** 
     * Constructor.
     *
     * @param store the backing store to use as the cache store.
     */
    protected DynamicMetadataCache(@Nonnull final DynamicBackingStore<IdentifierType, MetadataType> store) {
        this(store, null);
        
    }
    
    /**
     * Protected constructor. Used mainly for testing.
     *
     * @param store the backing store to use as the cache store.
     * @param executor override the executor service.
     */
    protected DynamicMetadataCache(@Nonnull final DynamicBackingStore<IdentifierType, MetadataType> store, 
            @Nullable final ScheduledExecutorService executor) { 
        super(store, executor);
        mgmtMappingFunction = id -> {
            final Instant now = Instant.now();
            final MetadataManagementData<IdentifierType> mgmt = new MetadataManagementData<>(id);
            mgmt.setRefreshTriggerTime(now.plus(maxCacheDuration));
            mgmt.setExpirationTime(now.plus(maxCacheDuration));
            return mgmt;
        };
        
    }
   
    
    /**
     *  Set the minimum cache duration for metadata.
     *  
     * @param duration the minimum cache duration
     */
    public void setMinCacheDuration(@Nonnull final Duration duration) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        Constraint.isNotNull(duration, "Duration cannot be null");
        Constraint.isFalse(duration.isNegative(), "Duration cannot be negative");
        
        minCacheDuration = duration;
    }
    
    /**
     *  Set the maximum cache duration for metadata.
     *  
     * @param duration the maximum cache duration
     */
    public void setMaxCacheDuration(@Nonnull final Duration duration) {        
        Constraint.isNotNull(duration, "Duration cannot be null");
        Constraint.isFalse(duration.isNegative(), "Duration cannot be negative");
        
        maxCacheDuration = duration;
    }
    
    /**
     * Set the metadata fetching strategy. 
     * 
     * @param strategy the strategy used to fetch metadata using a 'read-through' semantic.
     */
    public void setFetchStrategy(@Nonnull final Function<CriteriaSet, MetadataType> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        fetchStrategy = Constraint.isNotNull(strategy, "Dynamic Metadata fetch strategy can not be null");
    }
    
    /**
     * Set the initial cleanup task delay.
     * 
     * @param delay The initialCleanupTaskDelay to set.
     */
    public void setInitialCleanupTaskDelay(@Nonnull final Duration delay) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        Constraint.isNotNull(delay, "Cleanup task delay can not be null");
        Constraint.isFalse(delay.isNegative() || delay.isZero(), "Cleanup task delay must be positive");
        initialCleanupTaskDelay = delay;
        
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doDestroy() {      
        
        if (ratioGaugeFetchToGet != null) {
            MetricsSupport.remove(MetricRegistry.name(metricsBaseName, METRIC_RATIOGAUGE_FETCH_TO_GET), 
                    ratioGaugeFetchToGet);
        }
        if (gaugeNumLiveIndexedMetadata != null) {
            MetricsSupport.remove(MetricRegistry.name(metricsBaseName, METRIC_GAUGE_NUM_LIVE_INDEX_METADATA), 
                    gaugeNumLiveIndexedMetadata);
        }
        ratioGaugeFetchToGet = null;
        gaugeNumLiveIndexedMetadata = null;
        timerFetchFromSource = null;
        timerGet = null;
        
        super.doDestroy();
    }
   
    
    /**
     * Set the interval at which the cleanup task should run.
     * 
     * <p>Defaults to: 30 minutes.</p>
     * 
     * @param interval the interval to set
     */
    public void setCleanupTaskInterval(@Nonnull final Duration interval) {    
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        Constraint.isNotNull(interval, "Cleanup task interval may not be null");
        Constraint.isFalse(interval.isNegative() || interval.isZero(), "Cleanup task interval must be positive");
        
        cleanupTaskInterval = interval;
    }
    
    /**
     * Set the flag indicating whether idle entity data should be removed. 
     * 
     * @param flag true if idle entity data should be removed, false otherwise
     */
    public void setRemoveIdleEntityData(final boolean flag) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        removeIdleEntityData = flag;
    }
    
    /**
     * Set the maximum idle time for which the resolver will keep data for a given entityID, 
     * before it is removed.
     * 
     * <p>Defaults to: 8 hours.</p>
     * 
     * @param max the maximum entity data idle time
     */
    public void setMaxIdleEntityData(@Nonnull final Duration max) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        Constraint.isNotNull(max, "Max idle time cannot be null");
        Constraint.isFalse(max.isNegative(), "Max idle time cannot be negative");

        maxIdleEntityData = max;
    }
    
    /**
     * Set the metadata expiration time strategy.
     * 
     * @param strategy the strategy.
     */
    public void setMetadataExpirationTimeStrategy(
            @Nonnull final Function<ExpirationTimeContext<MetadataType>, Instant> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        metadataExpirationTimeStrategy = Constraint.isNotNull(strategy, "Metadata expiration strategy can not be null");
    }
    
    /**
     * Get the metadata expiration time strategy. 
     * 
     * @return the expiration time strategy.
     */
    @NonnullAfterInit 
    protected Function<ExpirationTimeContext<MetadataType>, Instant> getMetadataExpirationTimeStrategy() {
        return metadataExpirationTimeStrategy;
    }
    
    
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();  
        
        if (maxCacheDuration == null || minCacheDuration == null || maxIdleEntityData == null  || 
                cleanupTaskInterval == null || initialCleanupTaskDelay == null) {
            throw new ComponentInitializationException("Dynamic metadata cache not property initialized"); 
        }
        if (metadataExpirationTimeStrategy == null) {
            throw new ComponentInitializationException("Metadata expiration strategy can not be null");
        }
        if (fetchStrategy == null) {
            throw new ComponentInitializationException("Metadata fetching strategy can not be null");
        }
        
        initializeMetricsInstrumentation();
        
        getExecutorService().scheduleAtFixedRate(
                errorHandlingWrapper(new ExpiredAndIdleMetadataCleanupTask()), initialCleanupTaskDelay.toMillis(), 
                cleanupTaskInterval.toMillis(), TimeUnit.MILLISECONDS);
    }
    
    /**
     * Initialize the Metrics-based instrumentation.
     */
    private void initializeMetricsInstrumentation() {
       
        // initialization is synchronized at the top level
        if (metricsBaseName == null) {
            setMetricsBaseName(MetricRegistry.name(this.getClass(), getId()));
        }        
        
        final MetricRegistry metricRegistry = MetricsSupport.getMetricRegistry();
        if (metricRegistry != null) {
            timerGet = metricRegistry.timer(
                    MetricRegistry.name(metricsBaseName, METRIC_TIMER_GET));  
            timerFetchFromSource = metricRegistry.timer(
                    MetricRegistry.name(metricsBaseName, METRIC_TIMER_FETCH_FROM_ORIGIN_SOURCE));
            
            // Note that these gauges must use the support method to register in a synchronized fashion,
            // and also must store off the instances for later use in destroy.
            ratioGaugeFetchToGet = MetricsSupport.register(
                    MetricRegistry.name(metricsBaseName, METRIC_RATIOGAUGE_FETCH_TO_GET), 
                    new RatioGauge() {
                        @Override
                        protected Ratio getRatio() {
                            return Ratio.of(timerFetchFromSource.getCount(), 
                                    timerGet.getCount());
                        }},
                    true);
            
            gaugeNumLiveIndexedMetadata = MetricsSupport.register(
                    MetricRegistry.name(metricsBaseName, METRIC_GAUGE_NUM_LIVE_INDEX_METADATA),
                    () -> getBackingStore().getIndexedValues().keySet().size(),
                    true);
        }
    }
    
    
    /**
     * Set the base name for Metrics instrumentation.
     * 
     * @param baseName the Metrics base name
     */
    public synchronized void setMetricsBaseName(@Nullable final String baseName) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        metricsBaseName = StringSupport.trimOrNull(baseName);
    }
        
    /**
     * {@inheritDoc}
     * 
     * <p>Cast the backing store to the type used by this cache type.</p>
     */
    @Override @Nonnull protected DynamicBackingStore<IdentifierType, MetadataType> getBackingStore() {
        return (DynamicBackingStore<IdentifierType, MetadataType>) super.getBackingStore();
    }
    
    
    @Override
    @Nonnull @NonnullElements public List<MetadataType> get(
            @Nonnull @NotEmpty final CriteriaSet criteria) throws MetadataCacheException {
        
        if (!isInitialized()) {
            throw new MetadataCacheException("Metadata cache has not been initialized");
        }
        
        final Context contextResolve = MetricsSupport.startTimer(timerGet);
        
        try {
            final IdentifierType identifier = getCriteriaToIdentifierStrategy().apply(criteria);
            log.debug("{} Resolved criteria to identifier: {}", getLogPrefix(), identifier);
            
            if (identifier != null) {       
                //TODO check we can do this here, as another thread could change this concurrently?
                final MetadataManagementData<IdentifierType> mgmtData = getBackingStore()
                        .computeManagementDataIfAbsent(identifier, mgmtMappingFunction);
                
                // check metadata refresh is not needed before reading.
                List<MetadataType> allMetadata = Collections.emptyList();
                if (!shouldAttemptRefresh(mgmtData)) {
                    // TODO: Metadata that does not exist yet but its mgmtData has been created will attempt 
                    // a pointless read.
                    allMetadata = read(mgmtData, identifier);
                }
                if (allMetadata.isEmpty()) {
                    log.debug("Metadata for '{}' does not exist, is no longer valid, or is stale, "
                            + "attempting to fetch it", identifier);
                    fetch(mgmtData, identifier, criteria);
                    return read(mgmtData, identifier);
                } else {
                    log.debug("Metadata for '{}' found in cache", identifier);
                    return allMetadata;
                }   
            } else {
                //TODO: see SAML version, could resolve from criteria even if no identifier.
                log.debug("Identifier not resolvable from criteria, can not fetch metadata");
                return Collections.emptyList();
            }
        } finally {
            MetricsSupport.stopTimer(contextResolve);
        }
            
    }
    
    /**
     * Fetch metadata using the supplied fetch function, then save it to the
     * backing store.
     * 
     * @param mgmtData the metadata management data.
     * @param identifier the identifier of the metadata to fetch.
     * @param criteria the criteria used in determining how to fetch the metadata.
     * @throws MetadataCacheException on error.
     */
    private void fetch(@Nonnull final MetadataManagementData<IdentifierType> mgmtData,
            @Nonnull final IdentifierType identifier,
            @Nonnull @NotEmpty final CriteriaSet criteria) throws MetadataCacheException{
        
        final StampedLock sl = mgmtData.getStampLock();
        final long stamp = sl.writeLock();
        
        try {            
            if (!shouldAttemptRefresh(mgmtData)){
                // re-check another thread has not acquired this lock before hand - and therefore obtained
                // the metadata.
                final List<MetadataType> allMetadata = lookupIdentifier(identifier);
                if (!allMetadata.isEmpty()) {
                    log.debug("{} Metadata for '{}' was acquired while waiting for the write lock", 
                            getLogPrefix(), identifier);
                    return;                    
                }
            } else {
                log.trace("{} Metadata for '{}' is stale and requires refreshing", getLogPrefix(), identifier);
            }
            
            MetadataType resolvedMetadata = null;
            final Context contextFetchFromSource = MetricsSupport.startTimer(timerFetchFromSource);
            try {
                resolvedMetadata = fetchStrategy.apply(criteria);   
            } finally {
                MetricsSupport.stopTimer(contextFetchFromSource);
            }
            
            if (resolvedMetadata != null) {
                storeNewMetadata(mgmtData, resolvedMetadata, identifier);
            } else {
                log.warn("{} Metadata for '{}' could not be resolved from source", getLogPrefix(), identifier);
            }
            
        } finally {
            sl.unlock(stamp);
        } 
    }
    
    /**
     * Process the new metadata as follows:
     * <ol>
     * <li>Apply the metadata filtering strategy configured.</li>
     * <li>Check the fetched metadata's identifier matches that expected.</li>
     * <li>Write the metadata to the backing store (cache).</li>
     * <li>Update the metadata's management information with new refresh information.</li>
     * </ol>
     * @param mgmtData the management data.
     * @param metadata the new metadata to process.
     * @param expectedIdentifier the identifier expected on the new metadata.
     */
    private void storeNewMetadata(@Nonnull final MetadataManagementData<IdentifierType> mgmtData,
            @Nonnull final MetadataType metadata,
            @Nonnull final IdentifierType expectedIdentifier) {
        
        final MetadataType filteredMetadata = getMetadataFilterStrategy().apply(metadata, newFilterContext());
        
        if (filteredMetadata == null) {
            log.warn("{} Filtered metadata is null, no further processing performed", getLogPrefix());
            return;
        }
        
        final IdentifierType extractedIdentifier = getIdentifierExtractionStrategy().apply(filteredMetadata);
        
        if (extractedIdentifier == null) {
            log.warn("{} Metadata identifier could not be extracted, no further processing performed", getLogPrefix());
        }
        
        // equality method of the identifier is required to be implemented correctly.
        if (!Objects.equals(expectedIdentifier, extractedIdentifier)) {
            log.warn("{} New metadata's identifer '{}' does not match expected identifier '{}', will not process", 
                    getLogPrefix(), extractedIdentifier, expectedIdentifier); 
            return;
        }                   
        
        log.debug("{} Resolved metadata with identifier '{}'",getLogPrefix(), extractedIdentifier);
       
        writeToBackingStore(filteredMetadata);
        
        final Instant now = Instant.now();
        log.debug("{} For metadata '{}' expiration and refresh computation, 'now' is : {}", 
                getLogPrefix(), extractedIdentifier, now);
        
        mgmtData.setLastUpdateTime(now);
        
        mgmtData.setExpirationTime(getMetadataExpirationTimeStrategy()
                .apply(createExpirationTimeContext(filteredMetadata, now)));
        log.debug("{} Computed metadata '{}' expiration time: {}", getLogPrefix(), 
                extractedIdentifier, mgmtData.getExpirationTime());
        
        mgmtData.setRefreshTriggerTime(computeRefreshTriggerTime(mgmtData.getExpirationTime(), now));
        log.debug("{} Computed metadata '{}' refresh trigger time: {}", getLogPrefix(), 
                extractedIdentifier, mgmtData.getRefreshTriggerTime());
        
        log.info("{} Successfully loaded new Metadata with identifer '{}'", getLogPrefix(), extractedIdentifier);   
    }
    
    /**
     * Create an expiration time context from the given metadata, the min and max cache durations,
     * and the instant representing 'now'. 
     * 
     * @param metadata the metadata.
     * @param now the instant representing now.
     * 
     * @return an expiration time context.
     */
    protected ExpirationTimeContext<MetadataType> createExpirationTimeContext(
            @Nonnull final MetadataType metadata,
            @Nonnull final Instant now){
        return new ExpirationTimeContext<>(metadata, minCacheDuration, maxCacheDuration, now);
    }
    
    /**
     * Compute the refresh trigger time.
     * 
     * @param expirationTime the time at which the metadata effectively expires
     * @param nowDateTime the current date time instant
     * 
     * @return the time after which refresh attempt(s) should be made
     */
    @Nonnull private Instant computeRefreshTriggerTime(@Nullable final Instant expirationTime,
            @Nonnull final Instant nowDateTime) {
        
        final long now = nowDateTime.toEpochMilli();

        long expireInstant = 0;
        if (expirationTime != null) {
            expireInstant = expirationTime.toEpochMilli();
        }
        long refreshDelay = (long) ((expireInstant - now) * getRefreshDelayFactor());

        // if the expiration time was null or the calculated refresh delay was less than the floor
        // use the floor
        if (refreshDelay < minCacheDuration.toMillis()) {
            refreshDelay = minCacheDuration.toMillis();
        }

        return nowDateTime.plusMillis(refreshDelay);
    }
    
    /**
     * Read metadata from the cache under the lock relating to the Identifier of the metadata to find.
     *  
     * <p>The first read attempt is optimistic and occurs without acquiring a read lock. The optimistic
     * read is validated to ensure another thread has not acquired a write lock in the meantime. If it has,
     * a read lock is obtained and a further read is attempted - to ensure a consistent state. This
     * should improve efficiency given that metadata reads will out number metadata fetch/writes.</p>
     * 
     * @param mgmtData the metadata management data.
     * @param identifier the metadata identifier to use as a key to fetch.
     * 
     * @return a list of metadata that matches the given key, an empty list of none are found.
     * 
     * @throws MetadataCacheException on error.
     */
    @Nonnull private List<MetadataType> read(@Nonnull final MetadataManagementData<IdentifierType> mgmtData,
            @Nonnull final IdentifierType identifier) throws MetadataCacheException {
        
        // Do not read if expired.
        if (hasExpired(mgmtData)) {
            log.trace("{} Metadata has expired for '{}'", getLogPrefix(), identifier);
            return Collections.emptyList();
        }
        
        // record access attempt.
        mgmtData.recordEntityAccess();
        // get optimistic lock
        final StampedLock sl = mgmtData.getStampLock();
        long stamp = sl.tryOptimisticRead();
        
        // A single optimistic read. A write lock will break this, but we check for 
        // that using the validate method e.g. we keep track of locks, but we do not block a write lock
        // here. Most metadata operations will be read-only. Reads are not expensive, but writes (lookups) are.
        final List<MetadataType> allMetadata = lookupIdentifier(identifier);
        
        if (sl.validate(stamp)) {
            return allMetadata;
        } else {
            // OK, was tampered with, lets do it with a hard read lock.
            stamp = sl.readLock();            
            try {
                return lookupIdentifier(identifier);                  
            } finally {
                sl.unlock(stamp);
            }           
        }
        
    }

    
    /**
     * Cleanup task that removes expired and idle metadata from the backing store.
     */
    private class ExpiredAndIdleMetadataCleanupTask implements Runnable {
        
        /** Logger. */
        @Nonnull private final Logger log = LoggerFactory.getLogger(ExpiredAndIdleMetadataCleanupTask.class);

        @Override
        public void run() {
            if (isDestroyed() || !isInitialized()) {
                // just in case the metadata resolver was destroyed before this task runs, 
                // or if it somehow is being called on a non-successfully-inited cache instance.
                log.debug("BackingStoreCleanupSweeper will not run because: inited: {}, destroyed: {}",
                        isInitialized(), isDestroyed());
                return;
            }
            log.trace("{} Running metadata cleanup background timer task",getLogPrefix());
            removeExpiredAndIdleMetadata();
        }
        
        /**
         *  Purge metadata if {@link #isRemoveData(MetadataManagementData, Instant, Instant)} is true.
         */
        private void removeExpiredAndIdleMetadata() {
            final Instant now = Instant.now();
            final Instant earliestValidLastAccessed = now.minus(maxIdleEntityData);
            
            final DynamicBackingStore<IdentifierType, MetadataType> store = getBackingStore();
            final Set<IdentifierType> ids = new HashSet<>();
            ids.addAll(store.getIndexedValues().keySet());
            ids.addAll(store.getManagementDataIdentifiers());
            
            for (final IdentifierType identifier : ids) {
                final MetadataManagementData<IdentifierType> mgmtData = 
                        store.computeManagementDataIfAbsent(identifier, mgmtMappingFunction);
                
                final long stamp = mgmtData.getStampLock().writeLock();
                try {                                       
                    if (isRemoveData(mgmtData, now, earliestValidLastAccessed)) {
                        invalidate(identifier);
                        store.removeManagementData(identifier);
                    }                    
                } finally {
                    mgmtData.getStampLock().unlock(stamp);
                }
                
            }
            
        }
        
        /**
         * Determine whether metadata should be removed based on expiration and idle time data.
         * 
         * @param mgmtData the management data instance for the entity
         * @param now the current time
         * @param earliestValidLastAccessed the earliest last accessed time which would be valid
         * 
         * @return true if the entity is expired or exceeds the max idle time, false otherwise
         */
        private boolean isRemoveData(@Nonnull final MetadataManagementData<IdentifierType> mgmtData, 
                @Nonnull final Instant now, @Nonnull final Instant earliestValidLastAccessed) {
            if (removeIdleEntityData && mgmtData.getLastAccessedTime().isBefore(earliestValidLastAccessed)) {
                log.debug("{} Metadata exceeds maximum idle time, removing: {}", getLogPrefix(), mgmtData.getID());
                return true;
            } else if (now.isAfter(mgmtData.getExpirationTime())) {
                log.debug("{} Metadata has expired, removing: {}", getLogPrefix(), mgmtData.getID());
                return true;
            } else {
                return false;
            }
        }
        
    }


}
