/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.logic.Constraint;

/** 
 * Simple strategy for computing an expiry time by adding an expiry duration to the current time.
 */
public class DefaultSourceMetadataExpirationTimeStrategy 
                        implements Function<byte[], Instant> {
    
    /** How long after now should the metadata expire.*/
    @Nonnull private final Duration expiryDuration;
    
    /**
     * 
     * Constructor.
     *
     * @param duration the expiry duration.
     */
    public DefaultSourceMetadataExpirationTimeStrategy(@Nonnull final Duration duration) {
        expiryDuration = Constraint.isNotNull(duration, "Expiry duration can not be null");
    }

    @Override
    public Instant apply(@Nullable final byte[] sourceMetadata) {
        return Instant.now().plus(expiryDuration);
    }

}
