/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.type.MapType;

import net.shibboleth.utilities.java.support.logic.Constraint;

/** 
 * Deserializes a UTF-8 JSON string into a Map.
 * 
 * @param <V> the map entry type.
 */
public class DefaultJSONMapParsingStrategy<V extends Object> 
                        implements Function<byte[], List<Map<String, V>>> {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultJSONMapParsingStrategy.class);
    
    /** JSON object mapper. */
    @Nonnull private final ObjectMapper objectMapper;

    /** The map type used by object mapper for deserializing the JSON into desired format. */
    @Nonnull private final MapType mapType;
    
    /**
     * 
     * Constructor.
     *
     * @param mapper the object mapper to use.
     * @param valueClass the class for the map value items.
     */
    public DefaultJSONMapParsingStrategy(@Nonnull final ObjectMapper mapper, @Nonnull final Class<V> valueClass) {
        objectMapper = Constraint.isNotNull(mapper, "Object mapper can not be null");
        mapType = objectMapper.getTypeFactory().constructMapType(Map.class, String.class,
                Constraint.isNotNull(valueClass,"Value class can not be null"));
    }
    
    /**
     * 
     * Constructor.
     *
     * @param valueClass the class for the map value items.
     */
    public DefaultJSONMapParsingStrategy(@Nonnull final Class<V> valueClass) {
        this(new ObjectMapper(), valueClass);
    }

    /** {@inheritDoc} */
    @Override
    public List<Map<String, V>> apply(@Nullable final byte[] rawMetadata) {
        try {
            if (rawMetadata == null) {
                return Collections.emptyList();
            }
            final Map<String, V> parsed = 
                    objectMapper.readValue(new String(rawMetadata, StandardCharsets.UTF_8).replace("\\", "\\\\"),
                            mapType);
            if (parsed != null) {
                return List.of(parsed);
            }
            return Collections.emptyList();
        } catch (final JsonProcessingException e) {
            log.error("Could not parse input raw metadata to a map", e);
            return Collections.emptyList();
        }
    }
    
}
