/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Predicates;

import net.shibboleth.oidc.metadata.cache.LoadingStrategy;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * A specification for building a batch based read-ahead metadata cache.
 *
 * @param <IdentifierType> The metadata identifier type.
 * @param <MetadataType> the metadata type.
 */
public class BatchMetadataCacheBuilderSpec<IdentifierType, MetadataType> 
                        extends BaseMetadataCacheBuilderSpec<IdentifierType, MetadataType> 
                            implements MetadataCacheBuilderSpec<IdentifierType, MetadataType> {
    
    /** 
     * How to parse the loaded metadata from the loadingStrategy into a usable metadatatype.
     * Applicable for {@link BatchMetadataCache} types. 
     */
    @Nullable private Function<byte[], List<MetadataType>> parsingStrategy;
    
    /** The function to use to load metadata. Applicable for {@link BatchMetadataCache} types.*/
    @Nullable private LoadingStrategy loadingStrategy;
    
    /**
     * Refresh interval used when metadata does not contain any validUntil or cacheDuration information. 
     * Default value: 4 hours
     */
    @Nonnull @Positive private Duration maxRefreshDelay;

    /** Floor, in milliseconds, for the refresh interval. Default value: 5 minutes */
    @Nonnull @Positive private Duration minRefreshDelay;
    
    /** 
     * Is a match based on an identifier required? If not, 
     * all known metadata will be returned. Defaults to true - a match on identifier is required. 
     */
    @Nonnull private boolean matchOnIdentifierRequired;
    
    /** Determine the expiration time of the source batch loaded metadata.*/
    @Nullable private Function<byte[], Instant> sourceMetadataExpiryStrategy;
    
    /** Is the raw metadata bytes from the source valid? */
    @Nonnull private Predicate<byte[]> sourceMetadataValidPredicate;
    
    /** Constructor.*/
    public BatchMetadataCacheBuilderSpec() {
        super();
        maxRefreshDelay = Duration.ofHours(4);
        minRefreshDelay = Duration.ofMinutes(5);    
        matchOnIdentifierRequired = false;
        sourceMetadataValidPredicate = Predicates.alwaysTrue();
    }
    
    /**
     * Set the predicate which determines if the source metadata is valid or not.
     * 
     * @param predicate the predicate.
     */
    public void setSourceMetadataValidPredicate(@Nonnull final Predicate<byte[]> predicate) {        
       sourceMetadataValidPredicate = 
               Constraint.isNotNull(predicate, "Is source metadata valid predicate can not be null");
    }
    
    /**
     * Get the predicate which determines if the source metadata is valid or not.
     * 
     * @return the predicate.
     */
    @Nonnull protected Predicate<byte[]> getSourceMetadataValidPredicate() {
        return sourceMetadataValidPredicate;
    }
    
    
    /**
     * Set a strategy to find the metadata expiry date from a source metadata document as bytes.
     * 
     * @param strategy the strategy.
     */
    public void setSourceMetadataExpiryStrategy(@Nonnull final Function<byte[], Instant> strategy) {
        sourceMetadataExpiryStrategy = 
                Constraint.isNotNull(strategy, "Source metadata expiry strategy can not be null");
    }
    
    /**
     * Get the source metadata expiry date strategy.
     * 
     * @return the strategy.
     */
    @Nullable protected Function<byte[], Instant> getSourceMetadataExpiryStrategy() {
        return sourceMetadataExpiryStrategy;
    }
    
    
    /**
     * Set if a match on identifier is required in order to return results. If false and there are no
     * identifiers in the criteria to match on, all cached entries will be returned. 
     * 
     * @param required does the metadata lookup need to match the given criteria.
     */
    public void setMatchRequired(final boolean required) {
        matchOnIdentifierRequired = required;
    }
    
    /**
     * Get if an identifier criteria match is required.
     * 
     * @return is an identifier criteria match required.
     */
    public boolean isMatchOnIdentifierRequired(){
        return matchOnIdentifierRequired;
    }
    
    /**
     * Get the min delay to wait before refreshing metadata.
     * 
     * @return the min delay.
     */
    @Nonnull protected Duration getMinRefreshDelay() {
        return minRefreshDelay;
    }
    
    /**
     * Get the max delay to wait before refreshing metadata.
     * 
     * @return the max delay.
     */
    @Nonnull protected Duration getMaxRefreshDelay() {
        return maxRefreshDelay;
    }
    

    
    /**
     * Sets the minimum amount of time between refreshes.
     * 
     * @param delay minimum amount of time between refreshes
     */
    public void setMinRefreshDelay(@Positive @Nonnull final Duration delay) {
        Constraint.isFalse(delay == null || delay.isNegative(), "Minimum refresh delay must be greater than 0");
        minRefreshDelay = delay;
    }
    
    /**
     * Sets the maximum amount of time between refresh intervals.
     * 
     * @param delay maximum amount of time, in milliseconds, between refresh intervals
     */
    public void setMaxRefreshDelay(@Positive @Nonnull final Duration delay) {        
        Constraint.isFalse(delay == null || delay.isNegative(), "Maximum refresh delay must be greater than 0");
        maxRefreshDelay = delay;
    }    
    
    /**
     * Set he raw batch metadata to metadata type parsing strategy.
     * 
     * @param strategy the parsing strategy.
     */
    public void setParsingStrategy(@Nonnull final Function<byte[], List<MetadataType>> strategy) {
        parsingStrategy = Constraint.isNotNull(strategy, "Parsing strategy can not be null");
    }
    
    /**
     * Get the raw batch metadata to metadata type parsing strategy.
     * 
     * @return the parsing strategy.
     */
    @Nullable protected Function<byte[], List<MetadataType>> getParsingStrategy() {
        return parsingStrategy;
    }
    
    /**
     * Set the batch metadata loading strategy, applies to the  {@link BatchMetadataCache} type.
     * 
     * @param strategy the strategy to set.
     */
    public void setLoadingStrategy(final @Nonnull LoadingStrategy strategy) {
        loadingStrategy =  Constraint.isNotNull(strategy,"Batch metadata loading strategy can not be null");
    }
    
    /**
     * Get the batch metadata loading strategy. 
     * 
     * @return the strategy.
     */
    @Nullable protected LoadingStrategy getLoadingStrategy() {
        return loadingStrategy;
    }

}
