/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache.impl;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.metadata.BatchBackingStore;
import net.shibboleth.oidc.metadata.cache.CacheLoadingContext;
import net.shibboleth.oidc.metadata.cache.LoadingStrategy;
import net.shibboleth.oidc.metadata.cache.MetadataCache;
import net.shibboleth.oidc.metadata.cache.MetadataCacheException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A {@link MetadataCache metadata cache} implementation that supports 'refresh-ahead' semantics for batch 
 * cache updates. Does not support 'read-through' semantics if an entry does not exist in the cache.
 * 
 * <p> The metadata source could either be an aggregate with more than one metadata entry, or a single source
 * which contains a single metadata entry.</p>
 * 
 * <p>Supports the following:</p>
 * <ul>
 * <li>Batch reloading of metadata from the source using the supplied loading strategy. The cache 
 * is wiped and reloaded during each successful refresh cycle.</li>
 * </ul>
 * 
 * <p>The schedule for batch metadata reloads are based on:</p>
 * <ol>
 * <li>If no expiry is set on the metadata source, or one can not be computed, the max refresh delay is used.</li>
 * <li>If metadata expiry exists but is less than the min refresh delay, use the min refresh delay.</li>
 * <li>If metadata expiry exists and is greater than the min, use the metadata expiry.</li>
 * </ol>
 * 
 * <p>Does not support:</p>
 * <ol>
 * <li>Manual metadata refresh. Reloads only occur via the scheduled task. Supporting manual refresh
 * would require alterations to remove/cancel existing tasks.</li>
 * </ol>
 *
 * @param <IdentifierType> the metadata identifier type.
 * @param <MetadataType> the metadata type.
 */
@ThreadSafe
public class BatchMetadataCache<IdentifierType, MetadataType> 
                                extends AbstractMetadataCache<IdentifierType, MetadataType> {

    /** Class logger. */
    private final Logger log = LoggerFactory.getLogger(BatchMetadataCache.class);

    /**
     * Refresh interval used when metadata does not contain any validUntil or cacheDuration information.
     */
    @NonnullAfterInit @Positive private Duration maxRefreshDelay;

    /** Floor, in milliseconds, for the refresh interval. */
    @NonnullAfterInit @Positive private Duration minRefreshDelay;
    
    /** The function to use to load metadata.*/
    @NonnullAfterInit private LoadingStrategy loadingStrategy;
    
    /** How to parse the loaded metadata from the loadingStrategy into a usable metadatatype.*/
    @NonnullAfterInit private Function<byte[], List<MetadataType>> parsingStrategy;
    
    /** Determine the expiration time of the source batch loaded metadata.*/
    @NonnullAfterInit private Function<byte[], Instant> sourceMetadataExpiryStrategy;
    
    /** Is the raw metadata bytes from the source valid? */
    @NonnullAfterInit private Predicate<byte[]> sourceMetadataValidPredicate;
    
    /** 
     * Is a match based on an identifier required? If not, 
     * all known metadata will be returned. Defaults to true - a match on identifier is required. 
     */
    @Nonnull private boolean matchOnIdentifierRequired;
    
    /** A lock to use when reading from and loading the cache.*/
    @Nonnull private final ReadWriteLock readWriteLock; 
   

    /** 
     * Constructor.
     *
     * @param store the backing store.
     */
    protected BatchMetadataCache(@Nonnull final BatchBackingStore<IdentifierType, MetadataType> store) {
        this(store, null);
    }

    /**
     * 
     * Protected constructor.
     *
     * @param store the backing store.
     * @param executor the scheduled executor
     */
    protected BatchMetadataCache(@Nonnull final BatchBackingStore<IdentifierType, MetadataType> store,
            @Nullable final ScheduledExecutorService executor) {
        super(store, executor);
        matchOnIdentifierRequired = true;
        readWriteLock = new ReentrantReadWriteLock();
    }

    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        if (maxRefreshDelay == null || minRefreshDelay == null) {
            throw new ComponentInitializationException("Refreshable metadata cache not property initialized");
        }
        if (sourceMetadataExpiryStrategy == null) {
            throw new ComponentInitializationException("Source metadata expiration time strategy can not be null");
        }
        if (loadingStrategy == null) {
            throw new ComponentInitializationException("Loading strategy can not be null");
        }
        if (parsingStrategy == null) {
            throw new ComponentInitializationException("Parsing strategy can not be null");
        }
        if (sourceMetadataValidPredicate == null) {
            throw new ComponentInitializationException("Is source metadata valid predicate can not be null");
        }
   
        try {
            loadCache();
        } catch (final MetadataCacheException e) {
            throw new ComponentInitializationException("Error loading metadata during init", e);
        }      
    }
    
    /**
     * Set the predicate which determines if the source metadata is valid or not.
     * 
     * @param predicate the predicate.
     */
    public void setSourceMetadataValidPredicate(@Nonnull final Predicate<byte[]> predicate) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
       sourceMetadataValidPredicate = 
               Constraint.isNotNull(predicate, "Is source metadata valid predicate can not be null");
    }
    
    /**
     * Get the predicate which determines if the source metadata is valid or not.
     * 
     * @return the predicate.
     */
    @NonnullAfterInit protected Predicate<byte[]> getSourceMetadataValidPredicate() {
        return sourceMetadataValidPredicate;
    }
    
    /**
     * Set the strategy used to batch load metadata from a source.
     * 
     * @param strategy the loading strategy.
     */
    public void setLoadingStrategy(@Nonnull final LoadingStrategy strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        loadingStrategy = Constraint.isNotNull(strategy, "Loading strategy can not be null");
    }
    
    /**
     * Get the loading strategy.
     * 
     * @return the loading strategy
     */
    @NonnullAfterInit protected LoadingStrategy getLoadingStrategy() {
        return loadingStrategy;
    }
    
    /**
     * Set the  strategy used to convert raw metadata in bytes to the given metadata type.
     * 
     * @param strategy the parsing strategy.
     */
    public void setParsingStrategy(@Nonnull final Function<byte[], List<MetadataType>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        parsingStrategy = Constraint.isNotNull(strategy, "Parsing strategy can not be null");
    }
    
    /**
     * Get the parsing strategy.
     * 
     * @return the parsing strategy
     */
    @NonnullAfterInit protected Function<byte[], List<MetadataType>> getParsingStrategy() {
        return parsingStrategy;
    }
    
    /**
     * Set a strategy to find the metadata expiry date from a source metadata document as bytes.
     * 
     * @param strategy the strategy.
     */
    public void setSourceMetadataExpiryStrategy(@Nonnull final Function<byte[], Instant> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        sourceMetadataExpiryStrategy = 
                Constraint.isNotNull(strategy, "Source metadata expiry strategy can not be null");
    }
    
    /**
     * Get the source metadata expiry strategy.
     * 
     * @return the source metadata expiry strategy
     */
    @NonnullAfterInit protected Function<byte[], Instant> getSourceMetadataExpiryStrategy() {
        return sourceMetadataExpiryStrategy;
    }
        
    /**
     * Set if a match on identifier is required in order to return results. If false and there are no
     * identifiers in the criteria to match on, all cached entries will be returned. 
     * 
     * @param required does the metadata lookup need to match the given criteria.
     */
    public void setMatchOnIdentifierRequired(final boolean required) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);
        
        matchOnIdentifierRequired = required;
    }
    
    /**
     * Is a match on identifier required?
     * 
     * @return true if it is, false otherwise
     */
    protected boolean isMatchOnIdentifierRequired() {
        return matchOnIdentifierRequired;
    }
    
        
    /**
     * {@inheritDoc}
     * 
     * <p>Cast the backing store to the type used by this cache type.</p>
     */
    @Override @Nonnull protected BatchBackingStore<IdentifierType, MetadataType> getBackingStore() {
        return (BatchBackingStore<IdentifierType, MetadataType>) super.getBackingStore();
    }

    /**
     * Sets the minimum amount of time between refreshes.
     * 
     * @param delay minimum amount of time between refreshes
     */
    public void setMinRefreshDelay(@Positive @Nonnull final Duration delay) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        Constraint.isFalse(delay == null || delay.isNegative(), "Minimum refresh delay must be greater than 0");
        minRefreshDelay = delay;
    }
    
    /**
     * Get the minimum amount of time between refreshes.
     * 
     * @return the minimum refresh delay
     */
    @NonnullAfterInit protected Duration getMinRefreshDelay() {
        return minRefreshDelay;
    }

    /**
     * Sets the maximum amount of time between refresh intervals.
     * 
     * @param delay maximum amount of time, in milliseconds, between refresh intervals
     */
    public void setMaxRefreshDelay(@Positive @Nonnull final Duration delay) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        ComponentSupport.ifDestroyedThrowDestroyedComponentException(this);

        Constraint.isFalse(delay == null || delay.isNegative(), "Maximum refresh delay must be greater than 0");
        maxRefreshDelay = delay;
    }
    
    /**
     * Get the maximum amount of time between refresh intervals.
     * 
     * @return the delay maximum amount of time, in milliseconds, between refresh intervals
     */
    @NonnullAfterInit protected Duration getMaxRefreshDelay() {
        return maxRefreshDelay;
    }

    /**
     * {@inheritDoc}
     * 
     * <p>Acquires a read lock to prevent reading while the cache is loading.</p>
     */
    @Override @Nonnull @NonnullElements
    public List<MetadataType> get(@Nonnull final CriteriaSet criteria) throws MetadataCacheException {

        if (!isInitialized()) {
            throw new MetadataCacheException("Metadata cache has not been initialized");
        }
        
        readWriteLock.readLock().lock();
        
        try {      
            final IdentifierType identifier = getCriteriaToIdentifierStrategy().apply(criteria);
            log.debug("{} Resolved criteria to identifier: {}", getLogPrefix(), identifier);
    
            if (identifier != null) {
    
                final List<MetadataType> allMetadata = lookupIdentifier(identifier);
                if (allMetadata.isEmpty()) {
                    log.debug("{} No metadata candidates for '{}' found, returning empty result", 
                            getLogPrefix(), identifier);
                    return Collections.emptyList();
                } else {
                    log.debug("{} There are {} metadata candidates for '{}' found in cache", 
                            getLogPrefix(), allMetadata.size(), identifier);
                    return allMetadata;
                }
            } else if (!matchOnIdentifierRequired) { 
                log.debug("{} No identifier found to lookup, identifier match is not required, returning all known "
                        + "metadata",getLogPrefix());
                return Collections.unmodifiableList(getBackingStore().getOrderedValues());
            } else {            
                log.debug("{} No identifier found to lookup, returning empty result", getLogPrefix());
                return Collections.emptyList();
            }
        } finally {
            readWriteLock.readLock().unlock();
        }
        // TODO: see SAML version, could resolve from criteria from secondary index.
    }
    
    /**
     * Create a cache loading context based on the last refresh and last update time of the backing store.
     * 
     * @return the cache loading context
     */
    private CacheLoadingContext createLoadingContext() {
        return new CacheLoadingContext(getBackingStore().getLastUpdate(), getBackingStore().getLastRefresh());
    }
    
    /**
     * Reload the entire backing-store cache using the loading strategy.
     * 
     * <p>Acquires a write lock to prevent loading while the cache is being read from.</p>
     * 
     * @throws MetadataCacheException on loading error.
     */
    private void loadCache() throws MetadataCacheException {
        
        readWriteLock.writeLock().lock();
        try {
            log.debug("{} Populating metadata cache for '{}'",getLogPrefix(), loadingStrategy.getSourceIdentifier());
            final Instant now = Instant.now();
            Instant metadataExpiration = null;
            try {
                if (isDestroyed()) {
                    return;
                }
                
                // Any exception here is caught
                final byte[] rawFetchedMetadata = loadingStrategy.load(createLoadingContext());
                if (rawFetchedMetadata != null) {
                    if (sourceMetadataValidPredicate.test(rawFetchedMetadata)) {           
                        
                        final List<MetadataType> parsedMetadata = parsingStrategy.apply(rawFetchedMetadata);
                        if (parsedMetadata != null && !parsedMetadata.isEmpty()) {
                            log.info("{} Parsed {} metadata candidates, loading into cache", 
                                    getLogPrefix(), parsedMetadata.size());
                            freshLoad(parsedMetadata);
                            // Store away the original, raw, metadata bytes.
                            getBackingStore().setOriginalValue(rawFetchedMetadata);
                            // Set last update time, technically there is no guarantee the metadata was stored correctly
                            // at this point.
                            getBackingStore().setLastUpdate(now);
                        } 
                     
                        // Compute metadata expiration from whatever is in the cache (updated or not) will
                        // remain null if no cached original value.            
                        metadataExpiration = sourceMetadataExpiryStrategy.apply(getBackingStore().getOriginalValue());
                    } else {
                        // Metadata is not valid
                        log.warn("{} Source metadata is not valid, nothing to load", getLogPrefix());
                        //TODO do we do anything else here, or just let it tick over and try again 
                        // on the next refresh cycle
                    }
                } else {
                    log.info("{} Metadata has not changed since last refresh", getLogPrefix());
                }
                
            } catch (final Throwable t) {
                log.error("{} Error loading or parsing metadata",getLogPrefix(), t);
                if (t instanceof Exception) {
                    throw new MetadataCacheException((Exception) t);
                } else {
                    throw new MetadataCacheException(String.format("Saw an error of type '%s' with message '%s'", 
                            t.getClass().getName(), t.getMessage()));
                }
            } finally {            
                if (metadataExpiration == null || metadataExpiration.isBefore(now)) {
                    // Null, so forced to use max refresh delay.
                    scheduleNextRefresh(null);
                } else {
                    final Duration nextRefreshDelay = computeNextRefreshDelay(metadataExpiration);
                    scheduleNextRefresh(nextRefreshDelay);
                }
                // Set last attempted refresh even if failure.
                getBackingStore().setLastRefresh(now);
            }
        } finally {
            readWriteLock.writeLock().unlock();
        }
        
    }
    
    
    /**
     * Schedules the next refresh. If the given delay is 0 or null, then {@link #maxRefreshDelay} is used.
     * 
     * @param delay The delay before the next refresh.
     */
    private void scheduleNextRefresh(@Nullable final Duration delay) {
        Duration refreshDelay = delay;
        if (delay == null || delay.isZero()) {
            refreshDelay = maxRefreshDelay;
        }
        final Instant nextRefresh = Instant.now().plus(refreshDelay);
        final long nextRefreshDelay = nextRefresh.toEpochMilli() - System.currentTimeMillis();

        getExecutorService().schedule(errorHandlingWrapper(
                new AsynchronousRefreshAHeadTask()), nextRefreshDelay, TimeUnit.MILLISECONDS);
        
        log.info("{} Next refresh cycle for metadata provider '{}' will occur on '{}' ('{}' local time)",
                getLogPrefix(), loadingStrategy.getSourceIdentifier(), nextRefresh, 
                nextRefresh.atZone(ZoneId.systemDefault()));
    }
    
    
    /**
     * Computes the delay until the next refresh time based on the current metadata's expiration time and the refresh
     * interval floor.
     * 
     * @param expectedExpiration the time when the metadata is expected to expire and needs refreshing
     * 
     * @return delay until the next refresh time
     */
    @Nonnull private Duration computeNextRefreshDelay(final Instant expectedExpiration) {
        final long now = System.currentTimeMillis();

        long expireInstant = 0;
        if (expectedExpiration != null) {
            expireInstant = expectedExpiration.toEpochMilli();
        }
        long refreshDelay = (long) ((expireInstant - now) * getRefreshDelayFactor());

        // if the expiration time was null or the calculated refresh delay was less than the floor
        // use the floor
        if (refreshDelay < minRefreshDelay.toMillis()) {
            refreshDelay = minRefreshDelay.toMillis();
        }

        return Duration.ofMillis(refreshDelay);
    }

    /** A runnable that triggers a cache reload.*/
    private class AsynchronousRefreshAHeadTask implements Runnable {

        @Override
        public void run() {

            if (!isInitialized()) {
                // just in case the metadata provider was destroyed before this task runs
                return;
            }

            try {
                loadCache();
            } catch (final MetadataCacheException e) {
                log.warn("{} Failed to background re-load cache for '{}'",getLogPrefix(), 
                        loadingStrategy.getSourceIdentifier(), e);
                // nothing the thread can do.
                return;
            }

        }
        
        
    }

}
