/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


package net.shibboleth.oidc.metadata.cache.impl;

import java.util.List;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.Predicates;

import net.shibboleth.oidc.metadata.filter.MetadataFilterContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.annotation.constraint.Positive;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A based metadata cache builder specification.
 *
 * @param <IdentifierType> the identifier type.
 * @param <MetadataType> the metadata type.
 */
public abstract class BaseMetadataCacheBuilderSpec<IdentifierType, MetadataType> {
        
    /** Factor used to compute when the next refresh interval will occur. Default value: 0.75 */
    @Positive private Float refreshDelayFactor;   
          
    /** Strategy used to extract an identifier from the given metadata.*/
    @Nullable private Function<MetadataType, IdentifierType> identifierExtractionStrategy;
    
    
    /** Map criteria to identifiers to use as keys to the backing store.*/
    @Nullable private Function<CriteriaSet, IdentifierType> criteriaToIdentifierStrategy;
    
    /** A strategy to filter metadata. */
    @Nonnull private BiFunction<MetadataType, MetadataFilterContext , MetadataType> metadataFilterStrategy;
    
    /** An identifier to give the cache this specification builds. Used for logging.*/
    @Nonnull private String cacheId;
    
    /** 
     * A hook that is executed just before a cache entry will been removed/invalidated/evicted.
     * The metadata list could be {@literal null}, the identifier is never {@literal null}.
     */
    @Nullable private BiConsumer<List<MetadataType>, IdentifierType> metadataBeforeRemovalHook;    
    
    /** Is the metadata valid? Defaults to true. */
    @Nonnull private Predicate<MetadataType> metadataValidPredicate;
    
    /** Constructor.*/
    protected BaseMetadataCacheBuilderSpec() {
        // defaults       
        refreshDelayFactor = 0.75f;         
        cacheId = "Unknown";
        // create a default direct in/out filter
        metadataFilterStrategy = (metadata, context) -> metadata;  
        // create default TRUE is metadata valid predicate
        metadataValidPredicate = Predicates.alwaysTrue();

    }
    
    /**
     * Set the predicate which determines if a piece of metadata is valid or not.
     * 
     * @param predicate the predicate.
     */
    public void setMetadataValidPredicate(@Nonnull final Predicate<MetadataType> predicate) {
       metadataValidPredicate = Constraint.isNotNull(predicate, "Is metadata valid predicate can not be null");
    }
    
    /**
     * Get the predicate which determines if a piece of metadata is valid or not.
     * 
     * @return the predicate.
     */
    @Nonnull protected Predicate<MetadataType> getMetadataValidPredicate() {
        return metadataValidPredicate;
    }
    
    /**
     * Set the cache identifier of the cache this specification builds.
     * 
     * @param id the cache identifier.
     */
    public void setCacheId(@Nonnull @NotEmpty final String id) {
        cacheId = Constraint.isNotEmpty(id, "Cache identifier can not be null or empty");
    }
    
    /**
     * Get the cache identifier.
     * 
     * @return the cache identifier.
     */
    @Nonnull protected String getCacheId() {
        return cacheId;
    }
    
   
    /**
     * Set a hook to run before a metadata cache entry is removed from the cache.
     * <p>The hook is required to gracefully handle null metadata lists.</p>
     * 
     * @param hook the hook to run.
     */
    public void setMetadataBeforeRemovalHook(@Nullable final BiConsumer<List<MetadataType>, 
            IdentifierType> hook) {        
        metadataBeforeRemovalHook = hook;
    }
    
    /**
     * Get the metadata before removal hook. Could be {@literal null}.
     * 
     * @return the hook.
     */
    @Nullable protected BiConsumer<List<MetadataType>, IdentifierType> getMetadataBeforeRemovalHook() {
        return metadataBeforeRemovalHook;
    }
    
    /**
     * Set the metadata filtering strategy.
     * 
     * <p>Defaults to a no-op strategy.</p>
     * 
     * @param strategy the metadata filtering strategy.
     */
    public void setMetadataFilterStrategy(@Nonnull final BiFunction<MetadataType, MetadataFilterContext, 
            MetadataType> strategy) {  
        metadataFilterStrategy = Constraint.isNotNull(strategy,"Metadata filtering strategy can not be null");
    }
    
    /**
     * Get the metadata filter strategy.
     * 
     * @return the metadata filtering strategy
     */
    @Nonnull protected BiFunction<MetadataType, MetadataFilterContext, MetadataType> getMetadataFilterStrategy() {
        return metadataFilterStrategy;
    }

      
    /**
     * Set the identifier extraction strategy.
     * 
     * @param strategy the strategy to set.
     */
    public void setIdentifierExtractionStrategy(
            @Nonnull final Function<MetadataType, IdentifierType> strategy) {        
        identifierExtractionStrategy = Constraint.isNotNull(strategy, "Strategy can not be null");
    }
    

    /** 
     * Get the identifier extraction strategy.
     * 
     * @return strategy the strategy.
     */
    @Nullable protected Function<MetadataType, IdentifierType> getIdentifierExtractionStrategy() {
        return identifierExtractionStrategy;
    }
    
    /**
     * Set the criteria set to identifier lookup strategy.
     * 
     * @param strategy the strategy to set.
     */
    public void setCriteriaToIdentifierStrategy(@Nonnull final Function<CriteriaSet, IdentifierType> strategy) {
        criteriaToIdentifierStrategy =  
                Constraint.isNotNull(strategy,"Criteria to identifier strategy can not be null");
    }
    
    /**
     * Get the criteria set to identifier lookup strategy.
     * 
     * @return strategy the strategy.
     */
    @Nullable protected Function<CriteriaSet, IdentifierType> getCriteriaToIdentifierStrategy() {
        return criteriaToIdentifierStrategy;
    }
    
     
    /**
     * Gets the delay factor used to compute the next refresh time.
     * 
     * <p>Defaults to:  0.75.</p>
     * 
     * @return delay factor used to compute the next refresh time
     */
    @Nonnull protected Float getRefreshDelayFactor() {
        return refreshDelayFactor;
    }
    

    /**
     * Sets the delay factor used to compute the next refresh time. The delay must be between 0.0 and 1.0, exclusive.
     * 
     * <p>Defaults to:  0.75.</p>
     * 
     * @param factor delay factor used to compute the next refresh time
     */
    public void setRefreshDelayFactor(@Nonnull final Float factor) {

        if (factor <= 0 || factor >= 1) {
            throw new 
                ConstraintViolationException("Refresh delay factor must be a number between 0.0 and 1.0, exclusive");
        }

        refreshDelayFactor = factor;
    }

}
