/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.criterion;

import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;

import com.nimbusds.oauth2.sdk.id.Issuer;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.Criterion;

/**
 * A {@link Criterion} representing an OIDC issuer ID.
 */
@Immutable
public final class IssuerIDCriterion implements Criterion {
    
    /** The issuer ID. */
    @Nonnull @NotEmpty private final Issuer id;
    
    /**
     * 
     * Constructor.
     *
     * @param issuerId the id of the issuer, can not be null or empty
     */
    public IssuerIDCriterion(@Nonnull final Issuer issuerId) {
        id = Constraint.isNotNull(issuerId, "Issuer ID can not be null");
    }
    
    /**
     * Get the issuer ID.
     * 
     * @return the issuer ID. 
     */
    @Nonnull public Issuer getIssuerID() {
        return id;
    }
    
    

    /** {@inheritDoc} */
    @Override
    public String toString() {
        return "IssuerIDCriterion [id=" + id + "]";
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (getClass() != obj.getClass()) {
            return false;
        }
        final IssuerIDCriterion other = (IssuerIDCriterion) obj;
        return id.equals(other.id);
    }

    
    
    

}
