/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.criterion;

import javax.annotation.Nonnull;

import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.Criterion;

/**
 * A {@link Criterion} representing a reference to a client secret.
 */
public class ClientSecretReferenceCriterion implements Criterion {
    
    /** The client secret reference. */
    @Nonnull @NotEmpty private final String secretReference;
    
    /**
     * Constructor.
     *
     * @param reference The client secret reference.
     */
    public ClientSecretReferenceCriterion(@Nonnull @NotEmpty final String reference) {
        secretReference = Constraint.isNotEmpty(reference, "The client secret reference cannot be empty");
    }
    
    /**
     * Get the client secret reference.
     * 
     * @return The client secret reference.
     */
    @Nonnull @NotEmpty public String getSecretReference() {
        return secretReference;
    }
    
    /** {@inheritDoc} */
    @Override
    public String toString() {
        final StringBuilder builder = new StringBuilder();
        builder.append("ClientSecretReferenceCriterion [secretReference=");
        builder.append(secretReference);
        builder.append("]");
        return builder.toString();
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode() {
        return secretReference.hashCode();
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }

        if (obj == null) {
            return false;
        }

        if (obj instanceof String) {
            return secretReference.equals((String) obj);
        }

        return false;
    }


}
