/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache;

import java.util.List;

import javax.annotation.Nonnull;

import net.shibboleth.utilities.java.support.annotation.constraint.NonnullElements;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A cache for loading, storing, and retrieving metadata against against the supplied {@link CriteriaSet}.
 *  
 * <p>Implementations can choose the specific cache semantics implemented e.g. read-through, refresh-ahead etc.
 * and storage/eviction strategies.</p>
 * 
 * <p>Implementations are intended to be access concurrently, and therefore are required to be thread-safe.</p>
 *
 * @param <U> The metadata type.
 */
public interface MetadataCache<U> {
    
    /**
     * Get the list of metadata matching the given identifier. 
     * 
     * <p>The implementing method is required to be thread safe.</p>
     * 
     * @param criteria the criteria to use when getting the metadata from the cache or the source. 
     *  
     * @return a list of metadata matching the criteria.
     * 
     * @throws MetadataCacheException on error fetching metadata. 
     */
    @Nonnull @NonnullElements public List<U> get(@Nonnull @NotEmpty final CriteriaSet criteria)
            throws MetadataCacheException;

}
