/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata.cache;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

/** A strategy that loads binary information from a configured source.*/
public interface LoadingStrategy {
    
    /**
     * Get a friendly name source identifier to use in log statements.
     * 
     * @return the source identifier friendly name.
     */
    @Nonnull String getSourceIdentifier();
    
    /**
     * Load cache data from a source. If the source data has not
     * changed from the last time it was loaded (as specified in the cache loading context) a
     * {@literal null} should be returned. If source data can not be loaded for a known
     * and accepted reason, a {@literal null} should be returned. For any other error in 
     * loading cache data, the {@link CacheLoadingException} should be thrown.
     * 
     * @param context the cache loading context
     * 
     * @return the cache information as bytes.
     * 
     * @throws CacheLoadingException if there is an error loading the cache.
     */
    @Nullable byte[] load(@Nonnull final CacheLoadingContext context) throws CacheLoadingException;

}
