/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata;

import java.time.Instant;
import java.util.concurrent.locks.StampedLock;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.utilities.java.support.logic.Constraint;

/** 
 * Class that holds management data about an entities metadata.
 * 
 * @param <MetadataIdentifier> the key to the stored object. 
 * 
 */
//TODO removed negative lookup cache.
//TODO guard this class?
//TODO change to a generic ObjectManagmentData type - if we want to broaden beyond metadata.
public class MetadataManagementData<MetadataIdentifier> {
    
    /** The identifier of the entity managed by this instance. */
    @Nonnull private final MetadataIdentifier id;
    
    /** Last update time of the associated metadata. */
    @Nullable private Instant lastUpdateTime;
    
    /** Expiration time of the associated metadata. */
    private Instant expirationTime;
    
    /** Time at which should start attempting to refresh the metadata. */
    private Instant refreshTriggerTime;
    
    /** The last time at which the entity's backing store data was accessed. */
    private Instant lastAccessedTime;
    
    /** Read-write stamped lock which governs access to the metadata's backing store data. */
    private final StampedLock stmpLock;
    
    /** Constructor. 
     * 
     * @param identifier the entity ID managed by this instance
     */
    public MetadataManagementData(@Nonnull final MetadataIdentifier identifier) {
        id = Constraint.isNotNull(identifier, "ID was null");
        final Instant now = Instant.now();    
        lastAccessedTime = now;
        stmpLock = new StampedLock();
    }
    
    /**
     * Get the entity ID managed by this instance.
     * 
     * @return the entity ID
     */
    @Nonnull public MetadataIdentifier getID() {
        return id;
    }
    
    /**
     * Get the last update time of the metadata. 
     * 
     * @return the last update time, or null if no metadata is yet loaded for the entity
     */
    @Nullable public Instant getLastUpdateTime() {
        return lastUpdateTime;
    }

    /**
     * Set the last update time of the metadata.
     * 
     * @param dateTime the last update time
     */
    public void setLastUpdateTime(@Nonnull final Instant dateTime) {
        lastUpdateTime = dateTime;
    }
    
    /**
     * Get the expiration time of the metadata. 
     * 
     * @return the expiration time
     */
    @Nullable public Instant getExpirationTime() {
        return expirationTime;
    }

    /**
     * Set the expiration time of the metadata.
     * 
     * @param dateTime the new expiration time
     */
    public void setExpirationTime(@Nonnull final Instant dateTime) {
        expirationTime = Constraint.isNotNull(dateTime, "Expiration time may not be null");
    }
    
    /**
     * Get the refresh trigger time of the metadata. 
     * 
     * @return the refresh trigger time
     */
    @Nullable public Instant getRefreshTriggerTime() {
        return refreshTriggerTime;
    }

    /**
     * Set the refresh trigger time of the metadata.
     * 
     * @param dateTime the new refresh trigger time
     */
    public void setRefreshTriggerTime(@Nonnull final Instant dateTime) {
        refreshTriggerTime = Constraint.isNotNull(dateTime, "Refresh trigger time may not be null");
    }

    /**
     * Get the last time at which the entity's backing store data was accessed.
     * 
     * @return last access time
     */
    @Nullable public Instant getLastAccessedTime() {
        return lastAccessedTime;
    }
    
    /**
     * Record access of the entity's backing store data.
     */
    public void recordEntityAccess() {
        lastAccessedTime = Instant.now();
    }
    
    /**
     * Get the read-write lock instance which governs access to the metadata's backing store data. 
     * 
     * @return the lock instance
     */
    @Nonnull public StampedLock getStampLock() {
        return stmpLock;
    }

}
