/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.oidc.metadata;

import java.util.function.Predicate;

import javax.annotation.Nonnull;

import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.resolver.Criterion;

/**
 * Base class for all metadata criterion classes. If the correct object type this criterion accepts is not provided, 
 * {@link #test(Object)} returns ({@code defaultResultOnWrongType}). 
 *
 * @param <T> The metadata type this criterion accepts.
 */
public abstract class AbstractEvaluableMetadataCriterion<T> implements Predicate<T>, Criterion {
    
    /** Object type. */
    @Nonnull private final Class<T> objectType;
    
    /** What should the default return type be if the wrong type is supplied.*/
    @Nonnull private final boolean defaultResultOnWrongType;
    
    /**
     * 
     * Constructor.
     *
     * @param claz the class this criterion accepts.
     * @param defaultResult what should be returned if the criterion is not appropriate for the given type.
     */
    protected AbstractEvaluableMetadataCriterion(@Nonnull final Class<T> claz,
            @Nonnull final boolean defaultResult) {
        objectType = Constraint.isNotNull(claz, "Object type cannot be null");
        defaultResultOnWrongType = Constraint.isNotNull(defaultResult, "Default result on wrong type can not be null");
    }
    
    /**
     * Get the object type this criterion accepts.
     * 
     * @return the object type.
     */
    @Nonnull public Class<T> getType() {
        return objectType;
    }

    
    @Override
    public boolean test(final T t) {
        if (objectType.isInstance(t)) {
            return doTest(t);        
        }       
        return defaultResultOnWrongType;
    }
    
    /**
     * Evaluates the predicate on the given argument. Concrete implementations should override this method. 
     * 
     * @param metadata the metadata.
     * 
     * @return true if the predicate is true, false otherwise.
     */
    public abstract boolean doTest(final T metadata);
    
}