/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;
import net.shibboleth.idp.profile.logic.AbstractRelyingPartyPredicate;

/**
 * A predicate implementation that checks if attribute consent flag is enabled. The value is fetched from
 * {@link TokenClaimsSet#isConsentEnabled()} via
 * {@link OIDCAuthenticationResponseContext#getAuthorizationGrantClaimsSet()}
 * under outbound message context. Default value is false, if any of the objects in the chain is null.
 */
public class AttributeConsentEnabledInTokenClaimsSetPredicate extends AbstractRelyingPartyPredicate {
    
    /** {@inheritDoc} */
    public boolean test(@Nullable final ProfileRequestContext input) {
        final MessageContext outboundMessageCtx = input.getOutboundMessageContext();
        if (outboundMessageCtx != null) {
            final OIDCAuthenticationResponseContext oidcResponseContext = 
                    outboundMessageCtx.getSubcontext(OIDCAuthenticationResponseContext.class);
            if (oidcResponseContext != null && oidcResponseContext.getAuthorizationGrantClaimsSet() != null) {
                return oidcResponseContext.getAuthorizationGrantClaimsSet().isConsentEnabled();
            }
        }
        return false;
    }

}
