/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.time.Instant;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.opensaml.storage.RevocationCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientRegistrationRequest;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.storage.RevocationCacheContexts;
import net.shibboleth.idp.plugin.oidc.op.token.support.RegistrationClaimsSet;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.security.DataSealer;
import net.shibboleth.utilities.java.support.security.DataSealerException;

/**
 * Action that validates registration access token is a valid one. Token is valid if it is successfully unwrapped,
 * parsed as access token, is not expired, has not been revoked and contains relying party identifier. Validated
 * token is stored to under incoming message context {@link OIDCClientRegistrationTokenClaimsContext#getClaimsSet()}.
 */
public class ValidateRegistrationAccessToken extends AbstractOIDCRequestAction<OIDCClientRegistrationRequest> {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateRegistrationAccessToken.class);
    
    /** Strategy that will return a {@link RelyingPartyContext}. */
    @Nonnull private Function<ProfileRequestContext,RelyingPartyContext> relyingPartyContextLookupStrategy;
    
    /** Strategy to create or return a {@link OIDCClientRegistrationTokenClaimsContext}. */
    @Nonnull private Function<ProfileRequestContext,OIDCClientRegistrationTokenClaimsContext>
        registrationClaimsContextCreationStrategy;

    /** Data sealer for unwrapping authorization code. */
    @NonnullAfterInit private DataSealer dataSealer;

    /** Message revocation cache instance to use. */
    @NonnullAfterInit private RevocationCache revocationCache;
    
    /** JSON object mapper. */
    @NonnullAfterInit private ObjectMapper objectMapper;
    
    /** The relying party context to operate on. */
    @Nullable private RelyingPartyContext relyingPartyContext;

    /** The registration access token to be validated. */
    @Nullable @NotEmpty private String accessToken;
    
    /**
     * Constructor.
     */
    public ValidateRegistrationAccessToken() {
        relyingPartyContextLookupStrategy = new ChildContextLookup<>(RelyingPartyContext.class);
        registrationClaimsContextCreationStrategy = 
                new ChildContextLookup<>(OIDCClientRegistrationTokenClaimsContext.class, true).compose(
                        new InboundMessageContextLookup());
    }
    
    /**
     * Set the strategy used to return the {@link RelyingPartyContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setRelyingPartyContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, RelyingPartyContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        relyingPartyContextLookupStrategy =
                Constraint.isNotNull(strategy, "RelyingPartyContext lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to create or return the {@link OIDCClientRegistrationTokenClaimsContext}.
     * 
     * @param strategy creation strategy
     */
    public void setRegistrationClaimsContextCreationStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCClientRegistrationTokenClaimsContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        registrationClaimsContextCreationStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationTokenClaimsContext creation strategy cannot be null");
    }
    
    /**
     * Set the data sealer for handling access token.
     * 
     * @param sealer data sealer.
     */
    public void setSealer(@Nonnull final DataSealer sealer) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        dataSealer = Constraint.isNotNull(sealer, "Data sealer cannot be null");
    }
    
    /**
     * Set the JSON {@link ObjectMapper}.
     * 
     * @param mapper object mapper
     */
    public void setObjectMapper(@Nonnull final ObjectMapper mapper) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        objectMapper = Constraint.isNotNull(mapper, "Object mapper cannot be null");
    }

    /**
     * Set the revocation cache instance to use.
     * 
     * @param cache The revocationCache to set.
     */
    public void setRevocationCache(@Nonnull final RevocationCache cache) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        revocationCache = Constraint.isNotNull(cache, "RevocationCache cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (dataSealer == null) {
            throw new ComponentInitializationException("DataSealer cannot be null");
        }
        
        if (objectMapper == null) {
            throw new ComponentInitializationException("ObjectMapper cannot be null");
        }
        
        if (revocationCache == null) {
            throw new ComponentInitializationException("RevocationCache cannot be null");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        relyingPartyContext = relyingPartyContextLookupStrategy.apply(profileRequestContext);
        if (relyingPartyContext == null) {
            log.error("{} No relying party context found, invalid profile context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }

        if (getRequest().getAccessToken() == null || getRequest().getAccessToken().getValue() == null) {
            log.debug("{} No access token in the request, nothing to do", getLogPrefix());
            return false;
        }
        accessToken = getRequest().getAccessToken().getValue();
        
        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final RegistrationClaimsSet claimsSet;
        try {
            final String unwrapped = dataSealer.unwrap(accessToken);
            log.debug("{} Access token unwrapped into {}", getLogPrefix(), unwrapped);
            claimsSet = objectMapper.readValue(unwrapped, RegistrationClaimsSet.class);
        } catch (final DataSealerException | JsonProcessingException e) {
            log.error("{} Decoding access token failed: {}", getLogPrefix(), e);
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_GRANT);
            return;
        }

        log.debug("{} Registration access token decoded into {}", getLogPrefix(), claimsSet);

        if (Instant.now().isAfter(claimsSet.getExpiration())) {
            log.error("{} Registration access token exp is in the past {}", getLogPrefix(), claimsSet.getExpiration());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_GRANT);
            return;
        }
        if (revocationCache.isRevoked(RevocationCacheContexts.REGISTRATION_ACCESS_TOKEN, claimsSet.getJti())) {
            log.error("{} Registration access token {} has been revoked", getLogPrefix(),
                    claimsSet.getJti());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_GRANT);
            return;
        }
        final String relyingPartyId = claimsSet.getRelyingPartyId();
        if (relyingPartyId != null) {
            log.debug("{} Registration access token {} carries relying party identifier {}", getLogPrefix(),
                    claimsSet.getJti(), relyingPartyId);
            relyingPartyContext.setVerified(true);
            relyingPartyContext.setRelyingPartyId(relyingPartyId);
        } else {
            log.debug("{} Registration access token {} carries no relying party identifier", getLogPrefix(),
                    claimsSet.getJti());
        }
        
        log.debug("{} Registration access token {} successfully validated", getLogPrefix(), claimsSet.getJti());
        
        final OIDCClientRegistrationTokenClaimsContext registrationClaimsContext =
                registrationClaimsContextCreationStrategy.apply(profileRequestContext);
        if (registrationClaimsContext == null) {
            log.error("{} Registration token claims context could not be created, invalid profile context",
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return;
        }
        
        registrationClaimsContext.setClaimsSet(claimsSet);
    }

}