/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.net.URI;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestRedirectURILookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultValidRedirectUrisLookupFunction;
import net.shibboleth.oidc.profile.core.OidcEventIds;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that validates redirect uri is expected. Validated redirect uri is stored to response context.
 */
public class ValidateRedirectURI extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidateRedirectURI.class);

    /** Strategy used to obtain the redirect uri value in request. */
    @Nonnull private Function<ProfileRequestContext, URI> redirectURILookupStrategy;

    /** Strategy used to obtain the redirect uris to compare request value to. */
    @Nonnull private Function<ProfileRequestContext, Set<URI>> validRedirectURIsLookupStrategy;

    /** Strategy used to obtain registered redirect uris to compare if request had no redirect uri value. */
    @Nonnull private Function<ProfileRequestContext, Set<URI>> registeredRedirectURIsLookupStrategy;

    /** Whether to require redirect uri value in the request also when only single value is registered. */
    private boolean requireRequestedValue = true;

    /**
     * Constructor.
     */
    public ValidateRedirectURI() {
        redirectURILookupStrategy = new DefaultRequestRedirectURILookupFunction();
        validRedirectURIsLookupStrategy = new DefaultValidRedirectUrisLookupFunction();
        registeredRedirectURIsLookupStrategy = new DefaultValidRedirectUrisLookupFunction();
    }

    /**
     * Set the strategy used to locate the redirect uri of the request.
     * 
     * @param strategy lookup strategy
     */
    public void setRedirectURILookupStrategy(@Nonnull final Function<ProfileRequestContext, URI> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        redirectURILookupStrategy =
                Constraint.isNotNull(strategy, "RedirectURILookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the redirect uris to compare against.
     * 
     * @param strategy lookup strategy
     */
    public void setValidRedirectURIsLookupStrategy(@Nonnull final Function<ProfileRequestContext, Set<URI>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        validRedirectURIsLookupStrategy =
                Constraint.isNotNull(strategy, "ValidRedirectURIsLookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used  to obtain registered redirect uris to compare if request had no redirect uri value.
     * 
     * @param strategy lookup strategy
     */
    public void setRegisteredRedirectURIsLookupStrategy(@Nonnull final Function<ProfileRequestContext,
            Set<URI>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        registeredRedirectURIsLookupStrategy =
                Constraint.isNotNull(strategy, "RegisteredRedirectURIsLookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set whether to require redirect uri value in the request also when only single value is registered.
     * 
     * @param flag flag to set
     */
    public void setRequireRequestedValue(final boolean flag) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        requireRequestedValue = flag;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final URI requestRedirectURI = redirectURILookupStrategy.apply(profileRequestContext);

        final Set<URI> redirectionURIs = validRedirectURIsLookupStrategy.apply(profileRequestContext);
        if (redirectionURIs == null || redirectionURIs.isEmpty()) {
            log.warn("{} Client has not registered Redirection URIs. Redirection URI cannot be validated.",
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REDIRECT_URI);
            return;
        }

        if (requestRedirectURI == null) {
            handleNullRequestedURI(profileRequestContext, redirectionURIs);
            return;
        }

        if (redirectionURIs.contains(requestRedirectURI)) {
            getOidcResponseContext().setRedirectURI(requestRedirectURI);
            log.debug("{} Redirection URI validated {}", getLogPrefix(), requestRedirectURI);
            return;
        }

        String registered = "";
        for (final URI uri : redirectionURIs) {
            registered += registered.isEmpty() ? uri.toString() : ", " + uri.toString();
        }
        log.error("{} Redirection URI {} not matching any of the registered Redirection URIs [{}] ", getLogPrefix(),
                requestRedirectURI.toString(), registered);
        ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REDIRECT_URI);
    }

    /**
     * Handles the missing requested redirect URI case: it may be missing if it's not required to exist and the
     * registered and valid records contain only single matching value.
     * 
     * @param profileRequestContext profile request context
     * @param validRedirectionURIs set of valid redirection uris
     */
    protected void handleNullRequestedURI(final ProfileRequestContext profileRequestContext,
            @Nonnull @NotEmpty final Set<URI> validRedirectionURIs) {
        if (requireRequestedValue) {
            log.warn("{} Redirection URI of the request not located for verification", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REDIRECT_URI);
            return;
        }
        final Set<URI> registeredURIs = registeredRedirectURIsLookupStrategy.apply(profileRequestContext);
        if (registeredURIs == null || registeredURIs.isEmpty()) {
            log.warn("{} Client has not registered Redirection URIs. Redirection URI cannot be validated.",
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REDIRECT_URI);
            return;
        }
        if (registeredURIs.size() == 1 && validRedirectionURIs.size() == 1) {
            final URI singleRegisteredUri = registeredURIs.iterator().next();
            final URI singleValidUri = validRedirectionURIs.iterator().next();
            if (singleRegisteredUri.equals(singleValidUri)) {
                log.debug("{} No requested redirect_uri found, but allowing it due to single trusted value {}",
                        getLogPrefix(), singleRegisteredUri);
                getOidcResponseContext().setRedirectURI(singleRegisteredUri);
                return;
            } else {
                log.warn("{} Registered URI '{}' did not match with the valid one '{}'", getLogPrefix(),
                        singleRegisteredUri, singleValidUri);
                ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REDIRECT_URI);
                return;
            }
        }
        // more than one registered registered/valid URIs
        log.warn("{} Redirection URI of the request missing even though multiple values registered/valid",
                getLogPrefix());
        ActionSupport.buildEvent(profileRequestContext, OidcEventIds.INVALID_REDIRECT_URI);
    }
}