/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.storage.RevocationCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.openid.connect.sdk.OIDCScopeValue;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.profile.logic.DefaultTokenRevocationLifetimeLookupStrategy;
import net.shibboleth.idp.plugin.oidc.op.storage.RevocationCacheContexts;
import net.shibboleth.idp.plugin.oidc.op.token.support.AuthorizeCodeClaimsSet;
import net.shibboleth.idp.plugin.oidc.op.token.support.RefreshTokenClaimsSet;
import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.oidc.profile.config.logic.EnforceRefreshTokenRotationPredicate;
import net.shibboleth.oidc.profile.config.navigate.RefreshTokenClaimsSetManipulationStrategyLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.RefreshTokenLifetimeLookupFunction;

import org.opensaml.profile.action.ActionSupport;
import net.shibboleth.utilities.java.support.annotation.ParameterName;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.security.DataSealer;
import net.shibboleth.utilities.java.support.security.DataSealerException;
import net.shibboleth.utilities.java.support.security.IdentifierGenerationStrategy;
import net.shibboleth.utilities.java.support.security.impl.SecureRandomIdentifierGenerationStrategy;

/**
 * Action that creates a Refresh Token, and sets it to work context
 * {@link OIDCAuthenticationResponseContext#getRefreshToken()} located under
 * {@link ProfileRequestContext#getOutboundMessageContext()}. The refresh_token is created only if the request contains
 * offline_access - scope.
 */
public class SetRefreshTokenToResponseContext extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetRefreshTokenToResponseContext.class);

    /** Data sealer for handling access token. */
    @Nonnull private final DataSealer dataSealer;

    /** Message revocation cache instance to use. */
    @NonnullAfterInit private RevocationCache revocationCache;

    /** Strategy used to obtain the refresh token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> refreshTokenLifetimeLookupStrategy;

    /** Lookup function to supply strategy bi-function for manipulating token claims set. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        tokenClaimsSetManipulationStrategyLookupStrategy;

    /** The strategy used for manipulating the token claims set. */
    @Nullable private BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> manipulationStrategy;

    /** Strategy used to locate the {@link IdentifierGenerationStrategy} to use. */
    @Nonnull private Function<ProfileRequestContext,IdentifierGenerationStrategy> idGeneratorLookupStrategy;

    /** Strategy used to determine whether to revoke refresh tokens once they're used. */
    @Nonnull private Predicate<ProfileRequestContext> enforceRefreshTokenRotationCondition;

    /** Lookup function to supply token revocation lifetime. */
    @Nonnull private Function<JWTClaimsSet,Duration> tokenRevocationLifetimeLookupStrategy;

    /** Authorize Code / Refresh Token the refresh token will be based on. */
    @Nullable private TokenClaimsSet tokenClaimsSet;
    
    /** Refresh Token lifetime. */
    @Nullable private Duration refreshTokenLifetime;

    /** The generator to use. */
    @Nullable private IdentifierGenerationStrategy idGenerator;

    /**
     * Constructor.
     * 
     * @param sealer sealer to encrypt/hmac refresh token.
     */
    public SetRefreshTokenToResponseContext(@Nonnull @ParameterName(name = "sealer") final DataSealer sealer) {
        refreshTokenLifetimeLookupStrategy = new RefreshTokenLifetimeLookupFunction();
        dataSealer = Constraint.isNotNull(sealer, "DataSealer cannot be null");
        tokenClaimsSetManipulationStrategyLookupStrategy =
                new RefreshTokenClaimsSetManipulationStrategyLookupFunction();
        idGeneratorLookupStrategy = FunctionSupport.constant(new SecureRandomIdentifierGenerationStrategy());
        enforceRefreshTokenRotationCondition = new EnforceRefreshTokenRotationPredicate();
        tokenRevocationLifetimeLookupStrategy = new DefaultTokenRevocationLifetimeLookupStrategy();
    }

    /**
     * Set the revocation cache instance to use.
     * 
     * @param cache The revocationCache to set.
     */
    public void setRevocationCache(@Nonnull final RevocationCache cache) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        revocationCache = Constraint.isNotNull(cache, "RevocationCache cannot be null");
    }

    /**
     * Set the strategy used to obtain the access token lifetime.
     * 
     * @param strategy lookup strategy
     */
    public void setRefreshTokenLifetimeLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        refreshTokenLifetimeLookupStrategy =
                Constraint.isNotNull(strategy, "Refresh token lifetime lookup strategy cannot be null");
    }

    /**
     * Set the lookup function to supply strategy bi-function for manipulating token claims set.
     * 
     * @param strategy What to set
     */
    public void setTokenClaimsSetManipulationStrategyLookupStrategy(@Nonnull final
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        tokenClaimsSetManipulationStrategyLookupStrategy =
                Constraint.isNotNull(strategy, "Manipulation strategy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link IdentifierGenerationStrategy} to use.
     * 
     * @param strategy lookup strategy
     */
    public void setIdentifierGeneratorLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,IdentifierGenerationStrategy> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        idGeneratorLookupStrategy =
                Constraint.isNotNull(strategy, "Identifier generation strategy cannot be null");
    }

    /**
     * Set the condition used to determine whether to revoke refresh tokens once they're used.
     * 
     * @param condition condition to apply
     */
    public void setEnforceRefreshTokenRotationCondition(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        enforceRefreshTokenRotationCondition = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Set a lookup strategy for the token revocation lifetime.
     *
     * @param strategy What to set.
     */
    public void setTokenRevocationLifetimeLookupStrategy(
            @Nullable final Function<JWTClaimsSet,Duration> strategy) {
        tokenRevocationLifetimeLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (revocationCache == null) {
            throw new ComponentInitializationException("RevocationCache cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        if (getOidcResponseContext().getScope() == null ||
                !getOidcResponseContext().getScope().contains(OIDCScopeValue.OFFLINE_ACCESS)) {
            log.debug("{} No offline_access scope, nothing to do", getLogPrefix());
            return false;
        }
        
        refreshTokenLifetime = refreshTokenLifetimeLookupStrategy.apply(profileRequestContext);
        if (refreshTokenLifetime == null) {
            log.warn("{} No lifetime supplied for refresh token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }
        
        tokenClaimsSet = getOidcResponseContext().getAuthorizationGrantClaimsSet();
        if (tokenClaimsSet == null || !(tokenClaimsSet instanceof RefreshTokenClaimsSet)
                && !(tokenClaimsSet instanceof AuthorizeCodeClaimsSet)) {
            log.error("{} No token to base refresh on", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }

        manipulationStrategy = tokenClaimsSetManipulationStrategyLookupStrategy.apply(profileRequestContext);

        idGenerator = idGeneratorLookupStrategy.apply(profileRequestContext);
        if (idGenerator == null) {
            log.error("{} No identifier generation strategy", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }

        return true;
    }

 // Checkstyle: CyclomaticComplexity OFF

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final Instant dateExp = Instant.now().plus(refreshTokenLifetime);
        final String rootTokenId = StringSupport.trimOrNull(tokenClaimsSet.getRootTokenIdentifier()) == null ? 
                tokenClaimsSet.getID() : tokenClaimsSet.getRootTokenIdentifier();
        final RefreshTokenClaimsSet claimsSet =
                new RefreshTokenClaimsSet.Builder(tokenClaimsSet, Instant.now(), dateExp)
                .setJWTID(idGenerator)
                .setRootTokenIdentifier(rootTokenId)
                .build();
        
        if (manipulationStrategy != null) {
            log.debug("{} Manipulation strategy has been set, applying it to the claims set {}", getLogPrefix(),
                    claimsSet.serialize());
            final Map<String, Object> result = manipulationStrategy.apply(profileRequestContext,
                    claimsSet.getClaimsSet().toJSONObject());
            if (result == null) {
                log.debug("{} Manipulation strategy returned null, leaving token claims set untouched.",
                        getLogPrefix());
            } else {
                log.debug("{} Applying the manipulated claims into the token claims set", getLogPrefix());
                try {
                    claimsSet.setClaimsSet(JWTClaimsSet.parse(result));
                } catch (final ParseException e) {
                    log.error("{} The resulted claims set could not be transformed into ", getLogPrefix(), e);
                    ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
                    return;
                }
            }
        } else {
            log.debug("{} No manipulation strategy configured", getLogPrefix());
        }

        try {
            getOidcResponseContext().setRefreshToken(claimsSet.serialize(dataSealer));
            log.debug("{} Setting refresh token {} as {} to response context ", getLogPrefix(), claimsSet.serialize(),
                    getOidcResponseContext().getRefreshToken());
        } catch (final DataSealerException e) {
            log.error("{} Refresh Token generation failed {}", getLogPrefix(), e.getMessage());
            ActionSupport.buildEvent(profileRequestContext, EventIds.UNABLE_TO_ENCRYPT);
        }

        if (enforceRefreshTokenRotationCondition.test(profileRequestContext) && 
                tokenClaimsSet instanceof RefreshTokenClaimsSet) {
            final String jti = tokenClaimsSet.getID();
            final Duration lifetime = tokenRevocationLifetimeLookupStrategy.apply(tokenClaimsSet.getClaimsSet());
            if (lifetime == null || Duration.ZERO.equals(lifetime)) {
                log.error("{} Unable to fetch lifetime for the single token revocation", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
                return;                
            }
            log.debug("{} Revoking the refresh token {} used for issuing the new one", getLogPrefix(), jti);
            if (!revocationCache.revoke(RevocationCacheContexts.SINGLE_ACCESS_OR_REFRESH_TOKENS, jti, lifetime)) {
                log.error("{} Unable to store revocation into the revocation cache", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
                return;
            }
        }
    }

 // Checkstyle: CyclomaticComplexity ON

}