/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.cli;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.beust.jcommander.Parameter;

import net.shibboleth.idp.cli.AbstractCommandLineArguments;

/** Command line processing for issue-registration-access-token flow. */
public class IssueRegistrationAccessTokenArguments extends AbstractCommandLineArguments {
    
    /** The URL parameter name for the metadata policy location. */
    public static final String URL_PARAM_POLICY_LOCATION = "policyLocation";
    
    /** The URL parameter name for the access token lifetime. */
    public static final String URL_PARAM_LIFETIME = "tokenLifetime";
    
    /** The URL parameter name for the relying party identifier. */
    public static final String URL_PARAM_POLICY_ID = "policyId";

    /** The URL parameter name for the client identifier. */
    public static final String URL_PARAM_CLIENT_ID = "clientId";

    /** The URL parameter name for the replacement flag. */
    public static final String URL_PARAM_REPLACEMENT = "replacement";

    /** Metadata policy to embed in the token. */
    @Parameter(names = {"-loc", "--policyLocation"}, required = false, description = "Metadata policy location")
    @Nullable private String policyLocation;
    
    /** Lifetime for the access token to be issued. */
    @Parameter(names = {"-l", "--lifetime"}, required = false, description = "Lifetime for the access token")
    @Nullable private String lifetime;
    
    /** Metadata policy identifier for the access token to be issued. */
    @Parameter(names = {"-id", "--policyId"}, required = false, description = "Metadata policy ID")
    @Nullable private String policyId;

    /** Requested client identifier for the access token to be issued. */
    @Parameter(names = {"-c", "--clientId"}, required = false, description = "Client ID to request in the access token")
    @Nullable private String clientId;

    /** Flag to signal one-time use of the token. */
    @Parameter(names = {"-r", "--replacement"}, required = false,
            description = "Flag to request the ability to re-register the same client ID for the life of the token")
    @Nullable private boolean replacement;
    
    /** {@inheritDoc} */
    @Override
    public void validate() {
        if (policyLocation == null && policyId == null) {
            throw new IllegalArgumentException("One of policyLocation or policyId is required");
        } else if (clientId == null && replacement) {
            throw new IllegalArgumentException("Enabling replacement requires specifying a client ID");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected StringBuilder doBuildURL(@Nonnull final StringBuilder builder) {

        if (getPath() == null) {
            builder.append("/profile/admin/oidc/issue-registration-access-token");
        }

        try {
            if (policyLocation != null) {
                appendSeparator(builder)
                    .append(URL_PARAM_POLICY_LOCATION + "=")
                    .append(URLEncoder.encode(policyLocation, "UTF-8"));
            }
            
            if (policyId != null) {
                appendSeparator(builder)
                    .append(URL_PARAM_POLICY_ID + "=")
                    .append(URLEncoder.encode(policyId, "UTF-8"));
            }

            if (lifetime != null) {
                builder
                    .append("&" + URL_PARAM_LIFETIME + "=")
                    .append(URLEncoder.encode(lifetime, "UTF-8"));
            }
            
            if (clientId != null) {
                builder
                    .append("&" + URL_PARAM_CLIENT_ID + "=")
                    .append(URLEncoder.encode(clientId, "UTF-8"));
                
                if (replacement) {
                    builder.append("&" + URL_PARAM_REPLACEMENT + "=" + replacement);
                }
            }
        } catch (final UnsupportedEncodingException e) {
            // UTF-8 is a required encoding.
            throw new RuntimeException("URL encoding failed", e);
        }

        return builder;
     }
    
    /**
     * Append the proper parameter separator to the builder.
     * 
     * @param builder input builder
     * 
     * @return the input
     */
    @Nonnull private StringBuilder appendSeparator(@Nonnull final StringBuilder builder) {
        
        if (builder.toString().contains("?")) {
            builder.append('&');
        } else {
            builder.append('?');
        }
        
        return builder;
    }

}