/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.authn.impl;

import java.security.NoSuchAlgorithmException;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.security.auth.Subject;
import javax.security.auth.login.LoginException;

import net.shibboleth.idp.authn.AbstractUsernamePasswordCredentialValidator;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.authn.context.UsernamePasswordContext;
import net.shibboleth.idp.authn.principal.UsernamePrincipal;
import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.ThreadSafeAfterInit;
import net.shibboleth.utilities.java.support.codec.StringDigester;
import net.shibboleth.utilities.java.support.codec.StringDigester.OutputFormat;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.client.ClientMetadata;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;

/**
 * A password validator that authenticates against OIDC client metadata (which may itself be emulated
 * via SAML metadata).
 */
@ThreadSafeAfterInit
public class OIDCClientInfoCredentialValidator extends AbstractUsernamePasswordCredentialValidator {
    
    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(OIDCClientInfoCredentialValidator.class);
    
    /** Strategy that will return {@link OIDCMetadataContext}. */
    @Nonnull private Function<ProfileRequestContext,OIDCMetadataContext> oidcMetadataContextLookupStrategy;
    
    /** Digester for SHA-1. */
    @NonnullAfterInit private StringDigester digester;

    /** Client information. */
    @Nullable private OIDCClientInformation clientInformation = null;

    /** Constructor. */
    public OIDCClientInfoCredentialValidator() {
        oidcMetadataContextLookupStrategy = new ChildContextLookup<>(OIDCMetadataContext.class).compose(
                new InboundMessageContextLookup());
    }
    
    /**
     * Set the strategy used to return the {@link OIDCMetadataContext}.
     * 
     * @param strategy The lookup strategy.
     */
    public void setOidcMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCMetadataContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
    
        oidcMetadataContextLookupStrategy =
                Constraint.isNotNull(strategy, "OIDCMetadataContext lookup strategy cannot be null");
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();

        try {
            digester = new StringDigester("SHA-256", OutputFormat.BASE64);
        } catch (final NoSuchAlgorithmException e) {
            throw new ComponentInitializationException("Error creating digester", e);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected Subject doValidate(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nullable final WarningHandler warningHandler,
            @Nullable final ErrorHandler errorHandler) throws Exception {

        final OIDCMetadataContext oidcMetadataContext = oidcMetadataContextLookupStrategy.apply(profileRequestContext);
        if (oidcMetadataContext != null) {
            clientInformation = oidcMetadataContext.getClientInformation();
        }
        
        if (clientInformation == null || clientInformation.getOIDCMetadata() == null) {
            log.debug("{} OIDC client metadata is missing", getLogPrefix());
            return null;
        } else {
            final ClientMetadata clientMetadata = clientInformation.getMetadata();
            if (ClientAuthenticationMethod.NONE.equals(clientMetadata.getTokenEndpointAuthMethod())) {
                log.debug("{} OIDC client metadata contains 'none' type for endpoint authentication");
                final Subject subject = new Subject();
                subject.getPrincipals().add(
                        new UsernamePrincipal(applyTransforms(clientInformation.getID().getValue())));
                return super.populateSubject(subject);
            } else if (clientInformation.getSecret() == null) {
                log.debug("{} OIDC client metadata for '{}' missing client secret", getLogPrefix(),
                        clientInformation.getID());
                return null;
            }
        }

        return super.doValidate(profileRequestContext, authenticationContext, warningHandler, errorHandler);
    }

    /** {@inheritDoc} */
    @Override
    @Nullable protected Subject doValidate(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext,
            @Nonnull final UsernamePasswordContext usernamePasswordContext,
            @Nullable final WarningHandler warningHandler,
            @Nullable final ErrorHandler errorHandler) throws Exception {

        final String username = usernamePasswordContext.getTransformedUsername();
        log.debug("{} Attempting to authenticate effective client ID '{}' ", getLogPrefix(), username);
        
        final Secret secret = clientInformation.getSecret();
        if (secret.getValue().startsWith("{SHA2}")) {
            if (secret.getValue().substring(6).equals(digester.apply(usernamePasswordContext.getPassword()))) {
                log.info("{} Login by '{}' succeeded", getLogPrefix(), username);
                return populateSubject(new Subject(), usernamePasswordContext);
            }
        } else if (clientInformation.getSecret().getValue().equals(usernamePasswordContext.getPassword())) {
            log.info("{} Login by '{}' succeeded", getLogPrefix(), username);
            return populateSubject(new Subject(), usernamePasswordContext);
        }
        
        log.info("{} Login by '{}' failed", getLogPrefix(), username);
        
        final LoginException e = new LoginException(AuthnEventIds.INVALID_CREDENTIALS); 
        if (errorHandler != null) { 
            errorHandler.handleError(profileRequestContext, authenticationContext, e,
                    AuthnEventIds.INVALID_CREDENTIALS);
        }
        throw e;
    }

}