/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.authn.impl;

import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.AbstractOptionallyAuthenticatedRequest;
import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.auth.ClientSecretBasic;
import com.nimbusds.oauth2.sdk.auth.ClientSecretPost;
import com.nimbusds.oauth2.sdk.auth.TLSClientAuthentication;

import net.shibboleth.idp.authn.AbstractExtractionAction;
import net.shibboleth.idp.authn.AuthnEventIds;
import net.shibboleth.idp.authn.context.AuthenticationContext;
import net.shibboleth.idp.authn.context.CertificateContext;
import net.shibboleth.idp.authn.context.UsernamePasswordContext;
import net.shibboleth.oidc.authn.context.OAuth2ClientAuthenticationContext;
import net.shibboleth.oidc.profile.config.navigate.TokenEndpointAuthMethodLookupFunction;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;


/**
 * Extracts OAuth 2 client authentication details from a request and stores them in an
 * {@link OAuth2ClientAuthenticationContext} beneath the {@link AuthenticationContext} for subsequent
 * validation.
 * 
 * <p>Depending on the form of authentication, additional child contexts may be created to store
 * extracted credentials, and they may undergo configured transformations. For example, password-based
 * methods will result in a {@link UsernamePasswordContext}, certificate-based in an {@link CertificateContext},
 * etc.</p>
 * 
 * @pre ProfileRequestContext.getSubcontext(AuthenticationContext.class) != null
 * @post AuthenticationContext.getSubcontext(OAuth2ClientAuthenticationContext.class) ! null
 *  along with other contexts as appropriate
 * @event {@link EventIds#PROCEED_EVENT_ID}
 * @event {@link EventIds#INVALID_MSG_CTX}
 * @event {@link AuthnEventIds#NO_CREDENTIALS}
 */
public class ExtractClientAuthenticationFromRequest extends AbstractExtractionAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ExtractClientAuthenticationFromRequest.class);

    /** Lookup strategy for enabled client authentication methods. */
    @NonnullAfterInit private Function<ProfileRequestContext, Set<ClientAuthenticationMethod>>
        clientAuthMethodsLookupStrategy;

    /** Message to extract credentials from. */
    @Nullable private AbstractOptionallyAuthenticatedRequest request;

    /**
     * Constructor.
     */
    public ExtractClientAuthenticationFromRequest() {
        clientAuthMethodsLookupStrategy = new TokenEndpointAuthMethodLookupFunction();
    }

    /**
     * Set the lookup strategy for enabled client authentication methods.
     *
     * @param strategy What to set.
     */
    public void setClientAuthMethodsLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, Set<ClientAuthenticationMethod>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        clientAuthMethodsLookupStrategy = Constraint.isNotNull(strategy,
                "Client authentication methods lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {
        if (!super.doPreExecute(profileRequestContext, authenticationContext)) {
            return false;
        }
        
        if (profileRequestContext.getInboundMessageContext() != null) {
            final Object msg = profileRequestContext.getInboundMessageContext().getMessage();
            if (msg instanceof AbstractOptionallyAuthenticatedRequest) {
                request = (AbstractOptionallyAuthenticatedRequest) msg;
                return true;
            }
        }
        
        log.warn("{} Inbound message missing or of incorrect type", getLogPrefix());
        ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
        return false;
    }

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext,
            @Nonnull final AuthenticationContext authenticationContext) {

        final ClientAuthentication clientAuthentication = request.getClientAuthentication();
        
        final OAuth2ClientAuthenticationContext ctx =
                authenticationContext.getSubcontext(OAuth2ClientAuthenticationContext.class, true);
        ctx.setClientAuthentication(clientAuthentication);

        if (clientAuthentication == null) {
            log.debug("{} No OAuth client credentials in request", getLogPrefix());
            final Set<ClientAuthenticationMethod> methods =
                    clientAuthMethodsLookupStrategy.apply(profileRequestContext);
            //  Build event only if 'none' is not enabled in the profile configuration
            if (methods == null || !methods.contains(ClientAuthenticationMethod.NONE)) {
                ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            }
            return;
        }

        // Note the Nimbus APIs appear to prevent the client ID or secret from being null.
        
        if (ClientAuthenticationMethod.CLIENT_SECRET_BASIC.equals(clientAuthentication.getMethod())) {
            final ClientSecretBasic basic = (ClientSecretBasic) clientAuthentication;
            if (basic.getClientID() != null && basic.getClientSecret() != null) {
                final UsernamePasswordContext upContext = new UsernamePasswordContext();
                upContext.setUsername(applyTransforms(basic.getClientID().getValue()))
                    .setPassword(basic.getClientSecret().getValue());
                authenticationContext.addSubcontext(upContext, true);
            } else {
                log.warn("{} No OAuth client credentials in basic-auth request?", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            }
        } else if (ClientAuthenticationMethod.CLIENT_SECRET_POST.equals(clientAuthentication.getMethod())) {
            final ClientSecretPost post = (ClientSecretPost) clientAuthentication;
            if (post.getClientID() != null && post.getClientSecret() != null) {
                final UsernamePasswordContext upContext = new UsernamePasswordContext();
                upContext.setUsername(applyTransforms(post.getClientID().getValue()))
                    .setPassword(post.getClientSecret().getValue());
                authenticationContext.addSubcontext(upContext, true);
            } else {
                log.warn("{} No OAuth client credentials in POST request?", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, AuthnEventIds.NO_CREDENTIALS);
            }
        } else if (ClientAuthenticationMethod.TLS_CLIENT_AUTH.equals(clientAuthentication.getMethod()) ||
                ClientAuthenticationMethod.SELF_SIGNED_TLS_CLIENT_AUTH.equals(clientAuthentication.getMethod())) {
            final TLSClientAuthentication tls = (TLSClientAuthentication) clientAuthentication;
            if (tls.getClientX509Certificate() != null) {
                final CertificateContext certContext = new CertificateContext();
                certContext.setCertificate(tls.getClientX509Certificate());
                authenticationContext.addSubcontext(certContext, true);
            }
        }
    }
// Checkstyle: CyclomaticComplexity ON

}