/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.oidc.metadata.policy.MetadataPolicy;
import net.shibboleth.oidc.metadata.policy.impl.MetadataPolicyHelper;
import net.shibboleth.utilities.java.support.collection.Pair;
import net.shibboleth.utilities.java.support.logic.ConstraintViolationException;

/**
 * <p>A function that merges two maps of metadata policies according to the rules specified in the OIDC federation spec
 * (draft 17), section 5.1.3.1:</p>
 * <ul>
 * <li>
 * subset_of: The result of merging the values of two subset_of operators is the intersection of the operator values.
 * </li>
 * <li>
 * one_of: The result of merging the values of two one_of operators is the intersection of the operator values.
 * </li>
 * <li>
 * superset_of: The result of merging the values of two superset_of operators is the union of the operator values.
 * </li>
 * <li>
 * add: The result of merging the values of two add operators is the union of the values.
 * </li>
 * <li>
 * value: Merging two value operators is NOT allowed unless the two operator values are equal.
 * </li>
 * <li>
 * default: Merging two default operators is NOT allowed unless the two operator values are equal.
 * </li>
 * <li>
 * essential: If a superior has specified essential=true, then a subordinate cannot change that. If a superior has
 * specified essential=false, then a subordinate is allowed to change that to essential=true. If a superior has not
 * specified essential, then a subordinate can set essential to true or false.
 * </li>
 * </ul>
 * <p>The function returns a pair of map of merged metadata policies and a boolean indicating if the merging operators
 * in the policies were compliant.
 */
public class DefaultMetadataPolicyMergingStrategy implements
    BiFunction<Map<String, MetadataPolicy>, Map<String, MetadataPolicy>,
        Pair<Map<String, MetadataPolicy>, Boolean>> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(DefaultMetadataPolicyMergingStrategy.class);

    /** {@inheritDoc} */
    @Nonnull
    public Pair<Map<String, MetadataPolicy>, Boolean> apply(@Nullable final Map<String,MetadataPolicy> first,
            @Nullable final Map<String,MetadataPolicy> second) {
        if (first == null || first.isEmpty()) {
            return new Pair<>(second, Boolean.TRUE);
        } else if (second == null || second.isEmpty()) {
            return new Pair<>(first, Boolean.TRUE);
        }
        final Set<String> combinedKeys = Stream.concat(first.keySet().stream(), 
                second.keySet().stream()).collect(Collectors.toSet());
        final Map<String, MetadataPolicy> result = new HashMap<>();
        boolean valid = true;
        for (final String key : combinedKeys) {
            try {
                result.put(key, MetadataPolicyHelper.mergeMetadataPolicies(first.get(key), second.get(key)));
            } catch (final ConstraintViolationException e) {
                log.warn("Incompatible metadata policies for claim '{}' that cannot be merged.", key);
                valid = false;
            }
        }
        return new Pair<>(result, Boolean.valueOf(valid));
    }

}