/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.security.credential.Credential;
import org.opensaml.xmlsec.EncryptionConfiguration;
import org.opensaml.xmlsec.SignatureSigningConfiguration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKSet;

import net.minidev.json.JSONObject;
import net.shibboleth.idp.plugin.oidc.op.messaging.JSONSuccessResponse;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.config.SecurityConfiguration;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.profile.config.OIDCSecurityConfiguration;
import net.shibboleth.oidc.security.impl.CredentialConversionUtil;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action that forms outbound message containing keyset. Keys of the keyset are located from security configuration.
 * Formed message is set to {@link ProfileRequestContext#getOutboundMessageContext()}.
 */
public class FormOutboundKeySetResponseMessage extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(FormOutboundKeySetResponseMessage.class);

    /**
     * Strategy used to locate the {@link RelyingPartyContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext, RelyingPartyContext> relyingPartyContextLookupStrategy;

    /** Security configuration we look for keys to publish. */
    @Nullable private OIDCSecurityConfiguration secConfiguration;

    /** Constructor. */
    public FormOutboundKeySetResponseMessage() {
        relyingPartyContextLookupStrategy = new ChildContextLookup<>(RelyingPartyContext.class);
    }

    /**
     * Set the strategy used to locate the {@link RelyingPartyContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link RelyingPartyContext} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setRelyingPartyContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, RelyingPartyContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        relyingPartyContextLookupStrategy =
                Constraint.isNotNull(strategy, "RelyingPartyContext lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }

        final RelyingPartyContext rpCtx = relyingPartyContextLookupStrategy.apply(profileRequestContext);
        if (rpCtx == null) {
            log.debug("{} No relying party context associated with this profile request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_RELYING_PARTY_CTX);
            return false;
        }

        if (rpCtx.getProfileConfig() == null) {
            log.debug("{} No profile configuration associated with this profile request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_RELYING_PARTY_CTX);
            return false;
        }

        final SecurityConfiguration securityConfig =
                rpCtx.getProfileConfig().getSecurityConfiguration(profileRequestContext);
        
        if (!(securityConfig instanceof OIDCSecurityConfiguration)) {
            log.debug("{} No security configuration associated with the profile configuration of the profile request",
                    getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_SEC_CFG);
            return false;
        }
        
        secConfiguration = (OIDCSecurityConfiguration) securityConfig;
        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final List<JWK> publishList = new ArrayList<JWK>();
        final SignatureSigningConfiguration signingConfig = secConfiguration.getSignatureSigningConfiguration();
        if (signingConfig != null) {
            convertAndPublishToList(signingConfig.getSigningCredentials(), publishList);
        }
        final EncryptionConfiguration encryptionConfig = secConfiguration.getRequestObjectDecryptionConfiguration();
        if (encryptionConfig != null) {
            convertAndPublishToList(encryptionConfig.getKeyTransportEncryptionCredentials(), publishList);
        }
        final JWKSet keySet = new JWKSet(publishList);
        final JSONObject keySetJson = new JSONObject(keySet.toJSONObject());
        profileRequestContext.getOutboundMessageContext().setMessage(new JSONSuccessResponse(keySetJson));
    }
    
    /**
     * Converts the given credentials into JWK and adds all the successfully converted JWKs to the given list.
     * 
     * @param credentials The list of credentials to be converted to JWKs.
     * @param publishList The list where the successfully converted JWKs are put.
     */
    protected void convertAndPublishToList(final List<Credential> credentials, final List<JWK> publishList) {
        if (credentials != null) {
            for (final Credential credential : credentials) {
                final JWK jwk = CredentialConversionUtil.credentialToKey(credential);
                if (jwk != null) {
                    publishList.add(jwk);
                }
            }
        }
    }

}