/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.idp.profile.context.navigate.ResponderIdLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.AudienceRestrictionsLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.IDTokenLifetimeLookupFunction;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.id.Audience;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.oauth2.sdk.id.Subject;
import com.nimbusds.openid.connect.sdk.claims.IDTokenClaimsSet;

/**
 * Action that creates a {@link IDTokenClaimsSet} object shell, and sets it to work context
 * {@link OIDCAuthenticationResponseContext} located under {@link ProfileRequestContext#getOutboundMessageContext()}.
 */
public class AddIDTokenShell extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(AddIDTokenShell.class);

    /** Strategy used to obtain the response issuer value. */
    @Nonnull private Function<ProfileRequestContext, String> issuerLookupStrategy;
    
    /** Strategy used to obtain the audiences to add. */
    @Nonnull private Function<ProfileRequestContext,Collection<String>> audienceRestrictionsLookupStrategy;

    /** Strategy used to obtain the ID token lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> idTokenLifetimeLookupStrategy;
    
    /** EntityID to populate into Issuer element. */
    @Nullable private String issuerId;

    /** The RelyingPartyContext to operate on. */
    @Nullable private RelyingPartyContext rpCtx;
    
    /** Audiences to add. */
    @Nullable private Collection<String> audiences;
    
    /** ID token lifetime. */
    @Nullable private Duration lifetime;

    /** Constructor. */
    public AddIDTokenShell() {
        issuerLookupStrategy = new ResponderIdLookupFunction();
        audienceRestrictionsLookupStrategy = new AudienceRestrictionsLookupFunction();
        idTokenLifetimeLookupStrategy = new IDTokenLifetimeLookupFunction();
    }

    /**
     * Set the strategy used to locate the issuer value to use.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        issuerLookupStrategy = Constraint.isNotNull(strategy, "IssuerLookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to obtain the audience restrictions to apply.
     * 
     * @param strategy lookup strategy
     */
    public void setAudienceRestrictionsLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,Collection<String>> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        audienceRestrictionsLookupStrategy =
                Constraint.isNotNull(strategy, "Audience restriction lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to obtain the ID token lifetime.
     * 
     * @param strategy lookup strategy
     */
    public void setIDTokenLifetimeLookupStrategy(@Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        idTokenLifetimeLookupStrategy =
                Constraint.isNotNull(strategy, "ID token lifetime lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        issuerId = issuerLookupStrategy.apply(profileRequestContext);
        if (issuerId == null) {
            log.warn("{} No issuer returned for ID token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }
        
        audiences = audienceRestrictionsLookupStrategy.apply(profileRequestContext);
        if (audiences == null || audiences.isEmpty()) {
            log.warn("{} No audiences returned for ID token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }
        
        lifetime = idTokenLifetimeLookupStrategy.apply(profileRequestContext);
        if (lifetime == null) {
            log.warn("{} No lifetime supplied for ID token", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }

        return true;
    }

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        final List<Audience> aud = new ArrayList<>();
        for (final String audience : audiences) {
            aud.add(new Audience(audience));
        }
        final Date exp = Date.from(Instant.now().plus(lifetime)); 
        final IDTokenClaimsSet idToken = new IDTokenClaimsSet(new Issuer(issuerId),
                new Subject(getOidcResponseContext().getSubject()), aud, exp, new Date());
        log.debug("{} Setting ID token shell to response context {}", getLogPrefix(),
                idToken.toJSONObject().toJSONString());
        getOidcResponseContext().setIDToken(idToken);
    }

}