/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.profile.impl;

import java.text.ParseException;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.TokenIntrospectionSuccessResponse;
import com.nimbusds.oauth2.sdk.id.Audience;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.Issuer;
import com.nimbusds.oauth2.sdk.id.Subject;
import com.nimbusds.oauth2.sdk.token.AccessTokenType;

import net.shibboleth.idp.plugin.oidc.op.oauth2.messaging.context.OAuth2TokenMgmtResponseContext;
import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;
import net.shibboleth.idp.profile.AbstractProfileAction;

/**
 * Action that forms outbound token introspection success message. Formed message is set to
 * {@link ProfileRequestContext#getOutboundMessageContext()}.
 */
public class FormOutboundIntrospectionResponseMessage extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(FormOutboundIntrospectionResponseMessage.class);

    /** {@inheritDoc} */
    @Override protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        final OAuth2TokenMgmtResponseContext ctx =
                profileRequestContext.getOutboundMessageContext().getSubcontext(
                        OAuth2TokenMgmtResponseContext.class);
        
        if (ctx == null || ctx.getTokenClaimsSet() == null) {
            log.debug("{} Introspection of token failed, token was not valid", getLogPrefix());
            profileRequestContext.getOutboundMessageContext().setMessage(
                    new TokenIntrospectionSuccessResponse.Builder(false).build());
            return;
        }

        final JWTClaimsSet tokenClaimsSet = ctx.getTokenClaimsSet();

        try {
            String clientID = tokenClaimsSet.getStringClaim(TokenClaimsSet.KEY_CLIENTID);
            if (clientID == null) {
                clientID = tokenClaimsSet.getStringClaim(TokenClaimsSet.KEY_LEGACY_CLIENTID);
            }

            final TokenIntrospectionSuccessResponse.Builder builder =
                    new TokenIntrospectionSuccessResponse.Builder(true)
                        .clientID(new ClientID(clientID))
                        .tokenType(AccessTokenType.BEARER)
                        .expirationTime(tokenClaimsSet.getExpirationTime())
                        .issueTime(tokenClaimsSet.getIssueTime())
                        .subject(new Subject(tokenClaimsSet.getSubject()))
                        .issuer(new Issuer(tokenClaimsSet.getIssuer()));
            
            String claim = tokenClaimsSet.getStringClaim(TokenClaimsSet.KEY_USER_PRINCIPAL);
            if (claim != null) {
                builder.username(claim);
            }
            
            claim = tokenClaimsSet.getStringClaim(TokenClaimsSet.KEY_SCOPE);
            if (claim != null) {
                builder.scope(Scope.parse(claim));
            }
            
            if (!tokenClaimsSet.getAudience().isEmpty()) {
                builder.audience(tokenClaimsSet.getAudience()
                        .stream()
                        .map(Audience::new)
                        .collect(Collectors.toUnmodifiableList()));

            }
            
            profileRequestContext.getOutboundMessageContext().setMessage(builder.build());
            
        } catch (final ParseException e) {
            log.error("{} Failure extracting claims for response", getLogPrefix(), e);
            profileRequestContext.getOutboundMessageContext().setMessage(
                    new TokenIntrospectionSuccessResponse.Builder(false).build());
        }
    }

}