/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.oauth2.decoding.impl;

import java.io.IOException;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.decoder.MessageDecodingException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.MoreObjects;
import com.nimbusds.oauth2.sdk.TokenIntrospectionRequest;
import com.nimbusds.oauth2.sdk.http.HTTPRequest;
import com.nimbusds.oauth2.sdk.http.ServletUtils;

import net.shibboleth.idp.plugin.oidc.op.decoding.impl.RequestUtil;

/**
 * Message decoder decoding OpenID Connect {@link TokenIntrospectionRequest}s.
 */
public class OAuth2IntrospectionRequestDecoder extends BaseOAuth2RequestDecoder<TokenIntrospectionRequest> {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(OAuth2IntrospectionRequestDecoder.class);

    /** {@inheritDoc} */
    @Override
    protected TokenIntrospectionRequest parseMessage() throws MessageDecodingException {
        try {
            final HTTPRequest httpReq = ServletUtils.createHTTPRequest(getHttpServletRequest());
            getProtocolMessageLog().trace("Inbound request {}", RequestUtil.toString(httpReq));
            return TokenIntrospectionRequest.parse(httpReq);
        } catch (final com.nimbusds.oauth2.sdk.ParseException | IOException e) {
            log.error("Unable to decode inbound request: {}", e.getMessage());
            throw new MessageDecodingException(e);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected String getMessageToLog(final @Nullable TokenIntrospectionRequest message) {
        return message == null ? null : MoreObjects.toStringHelper(this).omitNullValues()
                .add("token", message.getToken())
                .add("customParameters", message.getCustomParameters())
                .add("clientAuthentication", RequestUtil.getClientAuthenticationLog(message.getClientAuthentication()))
                .add("clientAuthorization", RequestUtil.getAccessTokenLog(message.getClientAuthorization()))
                .add("endpointURI", message.getEndpointURI())
                .toString();
    }

}