/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.text.ParseException;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;

import net.shibboleth.idp.plugin.oidc.op.token.support.TokenClaimsSet;
import net.shibboleth.utilities.java.support.primitive.StringSupport;

/**
 * Default lookup function for fetching the root token identifier from the given claims set.
 */
public class DefaultRootTokenIdentifierLookupStrategy implements Function<JWTClaimsSet, String> {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(DefaultRootTokenIdentifierLookupStrategy.class);

    /** {@inheritDoc} */
    @Override
    @Nullable
    public String apply(@Nullable final JWTClaimsSet claimsSet) {
        if (claimsSet == null) {
            log.error("The given claims set was null, returning null");
            return null;
        }
        try {
            final String rootJti = claimsSet.getStringClaim(TokenClaimsSet.KEY_ROOT_JTI);
            if (StringSupport.trimOrNull(rootJti) != null) {
                log.debug("Root token identifier found from the claims set");
                return rootJti;
            }
        } catch (final ParseException e) {
            log.error("Could not parse the root token identifier from the claims set", e);
        }
        log.debug("Could not find root token identifier, returning null");
        return null;
    }
    
}