/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.shibboleth.idp.plugin.oidc.op.cli.IssueRegistrationAccessTokenArguments;
import net.shibboleth.idp.profile.function.SpringFlowScopeLookupFunction;
import net.shibboleth.oidc.metadata.criterion.ResourceLocationCriterion;
import net.shibboleth.utilities.java.support.primitive.StringSupport;
import net.shibboleth.utilities.java.support.resolver.CriteriaSet;

/**
 * A function returning a {@link CriteriaSet} which contains the metadata policy document location as {@link
 * ResourceLocationCriterion}.
 * 
 * <p>The value is fetched from the SWF flow scope.</p>
 */
public class DefaultMetadataPolicyCriteriaLookupFunction implements Function<ProfileRequestContext, CriteriaSet> {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(DefaultMetadataPolicyCriteriaLookupFunction.class);

    /** {@inheritDoc} */
    @Override @Nullable
    public CriteriaSet apply(@Nonnull final ProfileRequestContext profileRequestContext) {
        final String location = new SpringFlowScopeLookupFunction(
                IssueRegistrationAccessTokenArguments.URL_PARAM_POLICY_LOCATION).apply(profileRequestContext);
        if (StringSupport.trimOrNull(location) == null) {
            log.debug("Could not find a policy location for building the criteria set, returning null");
            return null;
        }
        log.trace("Found a location {} to be included in the criteria set", location);
        final ResourceLocationCriterion criterion = new ResourceLocationCriterion(location);
        return new CriteriaSet(criterion);
    }

}