/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.logic;

import java.util.Collection;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.minidev.json.JSONObject;
import net.shibboleth.idp.attribute.AttributeEncodingException;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.context.AttributeContext;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoder;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscoderSupport;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.oidc.attribute.transcoding.OIDCAttributeTranscoder;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.annotation.constraint.NotEmpty;
import net.shibboleth.utilities.java.support.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.service.ReloadableService;
import net.shibboleth.utilities.java.support.service.ServiceableComponent;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.messaging.context.navigate.ContextDataLookupFunction;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A function that returns subject identifier from filtered claims. Encoded value of the first attribute that would be
 * encoded as claim sub is returned.
 * 
 * TODO: This needs revisiting, it's not really a great way to configure this. Specifying the attribute(s) to pull from
 * would be the expected solution here. It may be warranted to just repurpose the IdP's NameID generation layer since
 * that's nominally type-invariant anyway, it can handle both SAML 1 and SAML 2 objects now. 
 */
public class AttributeResolutionSubjectLookupFunction extends AbstractIdentifiableInitializableComponent
        implements ContextDataLookupFunction<ProfileRequestContext, String> {

    /**
     * Name of the sub claim.
     */
    @Nonnull @NotEmpty private static final String SUB_CLAIM_NAME = "sub";
    
    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(AttributeResolutionSubjectLookupFunction.class);

    /**
     * Strategy used to locate the {@link AttributeContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext, AttributeContext> attributeContextLookupStrategy;

    /** Transcoder registry service object. */
    @NonnullAfterInit private ReloadableService<AttributeTranscoderRegistry> transcoderRegistry;

    /**
     * Constructor.
     */
    AttributeResolutionSubjectLookupFunction() {
        attributeContextLookupStrategy = new ChildContextLookup<>(AttributeContext.class).compose(
                new ChildContextLookup<>(RelyingPartyContext.class));
    }

    /**
     * Sets the registry of transcoding rules to apply to encode attributes.
     * 
     * @param registry registry service interface
     */
    public void setTranscoderRegistry(@Nonnull final ReloadableService<AttributeTranscoderRegistry> registry) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        transcoderRegistry = Constraint.isNotNull(registry, "AttributeTranscoderRegistry cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link AttributeContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link AttributeContext} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setAttributeContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, AttributeContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        attributeContextLookupStrategy =
                Constraint.isNotNull(strategy, "AttributeContext lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (transcoderRegistry == null) {
            throw new ComponentInitializationException("AttributeTranscoderRegistry cannot be null");
        }
    }
    
    // Checkstyle: CyclomaticComplexity OFF

    /** {@inheritDoc} */
    @Nullable public String apply(@Nullable final ProfileRequestContext input) {
        final AttributeContext attributeCtx = attributeContextLookupStrategy.apply(input);
        if (attributeCtx == null) {
            log.debug("No AttributeSubcontext available, nothing to do");
            return null;
        }
        
        ServiceableComponent<AttributeTranscoderRegistry> component = null;
        try {
            component = transcoderRegistry.getServiceableComponent();
            if (component == null) {
                log.error("Attribute transoding service unavailable");
                return null;
            }
            
            log.debug("Searching for 'sub' claim among encodeable attributes");
            
            for (final IdPAttribute attribute : attributeCtx.getIdPAttributes().values()) {
                if (attribute != null && !attribute.getValues().isEmpty()) {
                    final Collection<TranscodingRule> transcodingRules =
                            component.getComponent().getTranscodingRules(attribute, JSONObject.class);
                    if (transcodingRules.isEmpty()) {
                        continue;
                    }
                    
                    for (final TranscodingRule rule : transcodingRules) {
                        try {
                            if (!SUB_CLAIM_NAME.equals(rule.getOrDefault(
                                    OIDCAttributeTranscoder.PROP_NAME, String.class, attribute.getId()))) {
                                continue;
                            }
                            
                            log.debug("Found candidate rule for attribute '{}'", attribute.getId());
                            
                            final AttributeTranscoder<JSONObject> transcoder =
                                    TranscoderSupport.<JSONObject>getTranscoder(rule);
                            final JSONObject encodedAttribute =
                                    transcoder.encode(input, attribute, JSONObject.class, rule);
                            if (encodedAttribute != null) {
                                final String sub = encodedAttribute.getAsString(SUB_CLAIM_NAME);
                                if (sub != null) {
                                    log.debug("Attribute '{}' produced usable value for sub claim", attribute.getId());
                                    return sub;
                                }
                            }
                        } catch (final AttributeEncodingException e) {
                            log.warn("Unable to encode attribute '{}' as OIDC claim", attribute.getId(), e);
                        }
                    }
                }
            }
        } finally {
            if (null != component) {
                component.unpinComponent();
            }
        }

        log.warn("Unable to produce a viable 'sub' claim");
        return null;
    }
    
    // Checkstyle: CyclomaticComplexity ON

}