/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.nio.charset.Charset;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.nimbusds.jose.util.Base64URL;

import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestCodeVerifierLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.token.support.AuthorizeCodeClaimsSet;
import net.shibboleth.oidc.profile.config.logic.AllowPKCEPlainPredicate;
import net.shibboleth.oidc.profile.config.logic.ForcePKCEPredicate;

import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;

import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * Action performs PKCE (https://oauth.net/2/pkce/) validation. If authentication request contains code_challenge
 * parameter, token request when passing authorization code as grant must include code_verifier parameter. Profile
 * configuration may used to force using PKCE, by default it is optional. Profile configuration may be used to allow
 * plain PKCE, by default it is not allowed.
 */
public class ValidatePKCE extends AbstractOIDCResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(ValidatePKCE.class);

    /**
     * Strategy used to locate the PKCE Code Verifier value.
     */
    @Nonnull private Function<ProfileRequestContext,String> codeVerifierLookupStrategy;

    /** Strategy used to determine whether to require PKCE. */
    @Nonnull private Predicate<ProfileRequestContext> forcePKCECondition;

    /** Strategy used to determine whether to allow plaintext PKCE. */
    @Nonnull private Predicate<ProfileRequestContext> allowPKCEPlainCondition;

    /** Whether PKCE is mandatory. */
    private boolean forcePKCE;

    /** Whether plain PKCE is allowed. */
    private boolean plainPKCE;

    /** PKCE code challenge. */
    @Nullable private String codeChallenge;

    /** PKCE code verifier. */
    @Nullable private String codeVerifier;

    /**
     * Constructor.
     */
    public ValidatePKCE() {
        codeVerifierLookupStrategy = new DefaultRequestCodeVerifierLookupFunction();
        forcePKCECondition = new ForcePKCEPredicate();
        allowPKCEPlainCondition = new AllowPKCEPlainPredicate();
    }

    /**
     * Set the condition used to determine whether to require PKCE.
     * 
     * @param condition condition to apply
     */
    public void setForcePKCECondition(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        forcePKCECondition = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Set the condition used to determine whether to allow plaintext PKCE.
     * 
     * @param condition condition to apply
     */
    public void setAllowPKCEPlainCondition(@Nonnull final Predicate<ProfileRequestContext> condition) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        allowPKCEPlainCondition = Constraint.isNotNull(condition, "Condition cannot be null");
    }

    /**
     * Set the strategy used to locate the Code Verifier value.
     * 
     * @param strategy strategy used to locate the Code Verifier value
     */
    public void setCodeVerifierLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        codeVerifierLookupStrategy = Constraint.isNotNull(strategy, "CodeVerifier lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        if (getOidcResponseContext().getAuthorizationGrantClaimsSet() == null) {
            log.warn("{} No validated authorization grant claims set available", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        if (!AuthorizeCodeClaimsSet.VALUE_TYPE_AC.equals(
                getOidcResponseContext().getAuthorizationGrantClaimsSet().getType())) {
            log.debug("{} No authorization code presented, PKCE not applied, nothing to do", getLogPrefix());
            return false;
        }
        
        forcePKCE = forcePKCECondition.test(profileRequestContext);
        plainPKCE = allowPKCEPlainCondition.test(profileRequestContext);
        
        codeChallenge = getOidcResponseContext().getAuthorizationGrantClaimsSet().getCodeChallenge();
        // Checks whether PKCE needs to be validated.
        if ((codeChallenge == null || codeChallenge.isEmpty()) && !forcePKCE) {
            log.debug("{} No PKCE code challenge in request, nothing to do", getLogPrefix());
            return false;
        }
        codeVerifier = codeVerifierLookupStrategy.apply(profileRequestContext);
        return true;
    }

// Checkstyle: CyclomaticComplexity OFF
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (codeChallenge == null || codeChallenge.isEmpty()) {
            // The authorize-flow already verifies this via ValidateCodeChallenge, but double-checking
            log.warn(
                    "{} No PKCE code challenge presented in authentication request" +
                            " even though required to access token endpoint", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
            return;
        }
        if (codeVerifier == null || codeVerifier.isEmpty()) {
            log.warn("{} No PKCE code verifier for code challenge presented in token request", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
            return;
        }
        if (codeChallenge.startsWith("plain")) {
            if (!plainPKCE) {
                log.warn("{} Plain PKCE code challenge method not allowed", getLogPrefix());
                ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
                return;
            }
            final String codeChallengeValue = codeChallenge.substring("plain".length());
            if (!codeVerifier.equals(codeChallengeValue)) {
                log.warn("{} PKCE code challenge {} not matching code verifier {}", getLogPrefix(), codeChallengeValue,
                        codeVerifier);
                ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_AUTHN_ERROR);
                return;
            }
        } else if (codeChallenge.startsWith("S256")) {
            final String codeChallengeValue = codeChallenge.substring("S256".length());
            MessageDigest md = null;
            try {
                md = MessageDigest.getInstance("SHA-256");
            } catch (final NoSuchAlgorithmException e) {
                log.warn("{} PKCE S256 code challenge verification requires SHA-256", getLogPrefix(),
                        codeChallengeValue, codeVerifier);
                ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_AUTHN_ERROR);
                return;
            }
            final byte[] hash = md.digest(codeVerifier.getBytes(Charset.forName("utf-8")));
            final String codeChallengeComparisonValue = Base64URL.encode(hash).toString();
            if (!codeChallengeComparisonValue.equals(codeChallengeValue)) {
                log.warn("{} PKCE code challenge {} not matching code verifier {}({})", getLogPrefix(),
                        codeChallengeValue, codeVerifier, codeChallengeComparisonValue);
                ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_AUTHN_ERROR);
                return;
            }
        } else {
            log.warn("{} Unknown code challenge method", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MESSAGE);
            return;
        }
    }
// Checkstyle: CyclomaticComplexity ON
    
}