/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.time.Duration;
import java.time.Instant;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.oauth2.sdk.client.ClientInformation;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformation;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientInformationResponse;

import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCClientRegistrationTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultOIDCClientRegistrationTokenClaimsContextLookupFunction;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.oidc.metadata.ClientInformationManager;
import net.shibboleth.oidc.metadata.ClientInformationManagerException;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

/**
 * An action that stores the {@link ClientInformation} from the {@link OIDCClientRegistrationResponseContext} to the
 * associated {@link ClientInformationManager}.
 */
public class StoreClientInformation extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(StoreClientInformation.class);
    
    /** The client information manager used for storing the information. */
    @Nullable private ClientInformationManager clientInformationManager;
    
    /** Strategy to obtain registration validity period policy. */
    @Nullable private Function<ProfileRequestContext,Duration> registrationValidityPeriodStrategy;
    
    /** Strategy used to locate the {@link OIDCClientRegistrationTokenClaimsContext} associated with the request. */
    @Nonnull private Function<ProfileRequestContext,OIDCClientRegistrationTokenClaimsContext>
        registrationTokenContextLookupStrategy;
    
    /** The OIDCClientRegistrationTokenClaimsContext from which to optionally obtain client ID. */
    @Nullable private OIDCClientRegistrationTokenClaimsContext registrationTokenCtx;
    
    /** The response message. */
    @Nullable private OIDCClientInformationResponse response;
    
    /**
     * Strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a given 
     * {@link MessageContext}.
     */
    @Nonnull private Function<MessageContext,OIDCClientRegistrationResponseContext> oidcResponseContextLookupStrategy;

    /** Constructor. */
    public StoreClientInformation() {
        oidcResponseContextLookupStrategy = new ChildContextLookup<>(OIDCClientRegistrationResponseContext.class);
        registrationTokenContextLookupStrategy = new DefaultOIDCClientRegistrationTokenClaimsContextLookupFunction();
    }
    
    /**
     * Set strategy function to obtain registration validity period.
     * 
     * @param strategy The strategy function.
     */
    public void setRegistrationValidityPeriodStrategy(
            @Nullable final Function<ProfileRequestContext,Duration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        registrationValidityPeriodStrategy = strategy;
    }
    
    /**
     * Get the client information manager used for storing the information.
     * 
     * @return The client information manager used for storing the information
     */
    @NonnullAfterInit public ClientInformationManager getClientInformationManager() {
        return clientInformationManager;
    }
    
    /**
     * Set the client information manager used for storing the information.
     * @param manager The client information manager used for storing the information.
     */
    public void setClientInformationManager(@Nonnull final ClientInformationManager manager) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        clientInformationManager = Constraint.isNotNull(manager, "The client information manager cannot be null!");
    }
    
    /**
     * Set the strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a given
     * {@link MessageContext}.
     * 
     * @param strategy strategy used to locate the {@link OIDCClientRegistrationResponseContext} associated with a 
     *         given {@link MessageContext}
     */
    public void setOidcResponseContextLookupStrategy(
            @Nonnull final Function<MessageContext,OIDCClientRegistrationResponseContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        oidcResponseContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationResponseContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCClientRegistrationTokenClaimsContext} associated with a given
     * request.
     * 
     * @param strategy lookup strategy
     */
    public void setRegistrationTokenContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,OIDCClientRegistrationTokenClaimsContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        registrationTokenContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCClientRegistrationTokenClaimsContext lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (clientInformationManager == null) {
            throw new ComponentInitializationException("ClientInformationManager cannot be null");
        }
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        if (profileRequestContext.getOutboundMessageContext() == null) {
            log.error("{} Unable to locate outbound message context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        final Object message = profileRequestContext.getOutboundMessageContext().getMessage();
        if (message == null || !(message instanceof OIDCClientInformationResponse)) {
            log.error("{} Unable to locate outbound message", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        registrationTokenCtx = registrationTokenContextLookupStrategy.apply(profileRequestContext);
        if (registrationTokenCtx != null && registrationTokenCtx.getClaimsSet() == null) {
            registrationTokenCtx = null;
        }
        
        response = (OIDCClientInformationResponse) message;
        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final OIDCClientInformation clientInformation = response.getOIDCClientInformation();
        Duration lifetime = registrationValidityPeriodStrategy != null ?
                registrationValidityPeriodStrategy.apply(profileRequestContext) : null;
        
        final boolean replace = registrationTokenCtx != null ?
                registrationTokenCtx.getClaimsSet().isReplacement() : false;
        
        log.debug("{} Storing client information (replace = {})", getLogPrefix(), replace);
        
        try {
            if (lifetime != null && lifetime.isZero()) {
                log.debug("{} Registration won't expire, lifetime set to 0", getLogPrefix());
                clientInformationManager.storeClientInformation(clientInformation, null, replace);
            } else {
                if (lifetime == null) {
                    log.debug("{} No registration validity period supplied, using default", getLogPrefix());
                    lifetime = Duration.ofHours(24);
                }
                final Instant expiration = Instant.now().plus(lifetime);
                log.debug("{} Registration will expire on {}", getLogPrefix(), expiration);
                clientInformationManager.storeClientInformation(clientInformation, expiration, replace);               
            }
        } catch (final ClientInformationManagerException e) {
            log.error("{} Could not store the client information", getLogPrefix(), e);
            ActionSupport.buildEvent(profileRequestContext, EventIds.IO_ERROR);
            return;
        }
        log.info("{} Client information successfully stored for {}", getLogPrefix(), 
                clientInformation.getID().getValue());
    }
    
}