/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import java.text.ParseException;
import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.openid.connect.sdk.claims.ClaimsSet;
import net.minidev.json.JSONArray;
import net.shibboleth.idp.authn.context.SubjectContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseConsentContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseContext;
import net.shibboleth.idp.plugin.oidc.op.messaging.context.OIDCAuthenticationResponseTokenClaimsContext;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestCodeChallengeLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestCodeChallengeMethodLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.DefaultRequestNonceLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.profile.context.navigate.OIDCAuthenticationResponseContextLookupFunction;
import net.shibboleth.idp.plugin.oidc.op.token.support.AuthorizeCodeClaimsSet;
import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.context.navigate.ResponderIdLookupFunction;
import net.shibboleth.oidc.profile.config.logic.AttributeConsentFlowEnabledPredicate;
import net.shibboleth.oidc.profile.config.navigate.AuthorizationCodeClaimsSetManipulationStrategyLookupFunction;
import net.shibboleth.oidc.profile.config.navigate.AuthzCodeLifetimeLookupFunction;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import net.shibboleth.utilities.java.support.annotation.constraint.NonnullAfterInit;
import net.shibboleth.utilities.java.support.component.ComponentInitializationException;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;
import net.shibboleth.utilities.java.support.logic.FunctionSupport;
import net.shibboleth.utilities.java.support.security.DataSealer;
import net.shibboleth.utilities.java.support.security.DataSealerException;
import net.shibboleth.utilities.java.support.security.IdentifierGenerationStrategy;
import net.shibboleth.utilities.java.support.security.impl.SecureRandomIdentifierGenerationStrategy;

/**
 * Action that creates a Authorization Code, and sets it to work context
 * {@link OIDCAuthenticationResponseContext#getAuthorizationCode()} located under
 * {@link ProfileRequestContext#getOutboundMessageContext()}.
 */
public class SetAuthorizationCodeToResponseContext extends AbstractOIDCAuthenticationResponseAction {

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SetAuthorizationCodeToResponseContext.class);

    /** Strategy used to obtain the response issuer value. */
    @Nonnull private Function<ProfileRequestContext, String> issuerLookupStrategy;

    /** Data sealer for handling authorization code. */
    @NonnullAfterInit private DataSealer dataSealer;

    /** The generator to use. */
    @Nullable private IdentifierGenerationStrategy idGenerator;

    /** Strategy used to locate the {@link IdentifierGenerationStrategy} to use. */
    @Nonnull private Function<ProfileRequestContext, IdentifierGenerationStrategy> idGeneratorLookupStrategy;

    /** Strategy used to obtain the authz code lifetime. */
    @Nonnull private Function<ProfileRequestContext,Duration> authzCodeLifetimeLookupStrategy;

    /** Strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext}. */
    @Nonnull
    private Function<ProfileRequestContext, OIDCAuthenticationResponseTokenClaimsContext>
    tokenClaimsContextLookupStrategy;

    /** Strategy used to locate the {@link OIDCAuthenticationResponseConsentContext}. */
    @Nonnull
    private Function<ProfileRequestContext, OIDCAuthenticationResponseConsentContext> consentContextLookupStrategy;

    /** Predicate used to check if consent is enabled with a given {@link ProfileRequestContext}. */
    @Nonnull
    private Predicate<ProfileRequestContext> consentEnabledPredicate;

    /** Strategy used to locate the code challenge. */
    @Nonnull private Function<ProfileRequestContext, String> codeChallengeLookupStrategy;
    
    /** Strategy used to locate the code challenge method. */
    @Nonnull private Function<ProfileRequestContext, String> codeChallengeMethodLookupStrategy;

    /** Lookup function to supply strategy bi-function for manipulating token claims set. */ 
    @Nonnull
    private Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
        tokenClaimsSetManipulationStrategyLookupStrategy;

    /** The strategy used for manipulating the token claims set. */
    @Nullable private BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>> manipulationStrategy;

    /** Subject context. */
    @Nullable private SubjectContext subjectCtx;

    /** Authorization code lifetime. */
    @Nullable private Duration authzCodeLifetime;

    /** Code challenge and the code challenge method stored to authz code.*/
    @Nullable private String codeChallenge;

    /**
     * Constructor.
     */
    public SetAuthorizationCodeToResponseContext() {
        codeChallengeLookupStrategy = new DefaultRequestCodeChallengeLookupFunction();
        codeChallengeMethodLookupStrategy = new DefaultRequestCodeChallengeMethodLookupFunction();
        tokenClaimsContextLookupStrategy =
                new ChildContextLookup<>(OIDCAuthenticationResponseTokenClaimsContext.class).compose(
                        new OIDCAuthenticationResponseContextLookupFunction());
        consentContextLookupStrategy =
                new ChildContextLookup<>(OIDCAuthenticationResponseConsentContext.class).compose(
                        new OIDCAuthenticationResponseContextLookupFunction());
        authzCodeLifetimeLookupStrategy = new AuthzCodeLifetimeLookupFunction();
        issuerLookupStrategy = new ResponderIdLookupFunction();
        consentEnabledPredicate = new AttributeConsentFlowEnabledPredicate();
        idGeneratorLookupStrategy = FunctionSupport.constant(new SecureRandomIdentifierGenerationStrategy());
        tokenClaimsSetManipulationStrategyLookupStrategy =
                new AuthorizationCodeClaimsSetManipulationStrategyLookupFunction();
    }

    /**
     * Set the data sealer instance to use.
     * 
     * @param sealer data sealer to use
     */
    public void setDataSealer(@Nonnull final DataSealer sealer) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        dataSealer = Constraint.isNotNull(sealer, "DataSealer cannot be null");
    }

    /**
     * Set the strategy used to locate the Code Challenge of the request.
     * 
     * @param strategy lookup strategy
     */
    public void setCodeChallengeLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        codeChallengeLookupStrategy =
                Constraint.isNotNull(strategy, "Code challenge lookup strategy cannot be null");
    }
    
    /**
     * Set the strategy used to locate the Code Challenge Method of the request.
     * 
     * @param strategy lookup strategy
     */
    public void setCodeChallengeMethodLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        codeChallengeMethodLookupStrategy =
                Constraint.isNotNull(strategy, "Code challenge method lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setOIDCAuthenticationResponseTokenClaimsContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCAuthenticationResponseTokenClaimsContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        tokenClaimsContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCAuthenticationResponseTokenClaimsContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link OIDCAuthenticationResponseTokenClaimsContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setOIDCAuthenticationResponseConsentContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCAuthenticationResponseConsentContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        consentContextLookupStrategy = Constraint.isNotNull(strategy,
                "OIDCAuthenticationResponseConsentContext lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to obtain the authz code lifetime.
     * 
     * @param strategy lookup strategy
     */
    public void setAuthzCodeLifetimeLookupStrategy(@Nonnull final Function<ProfileRequestContext,Duration> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        
        authzCodeLifetimeLookupStrategy =
                Constraint.isNotNull(strategy, "Authz code lifetime lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link IdentifierGenerationStrategy} to use.
     * 
     * @param strategy lookup strategy
     */
    public void setIdentifierGeneratorLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, IdentifierGenerationStrategy> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        idGeneratorLookupStrategy =
                Constraint.isNotNull(strategy, "IdentifierGenerationStrategy lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the issuer value to use.
     * 
     * @param strategy lookup strategy
     */
    public void setIssuerLookupStrategy(@Nonnull final Function<ProfileRequestContext, String> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);
        issuerLookupStrategy = Constraint.isNotNull(strategy, "IssuerLookupStrategy lookup strategy cannot be null");
    }

    /**
     * Set the predicate used to check if consent is enabled with a given {@link ProfileRequestContext}.
     * 
     * @param predicate predicate used to check if consent is enabled with a given {@link ProfileRequestContext}.
     */
    public void setConsentEnabledPredicate(@Nonnull final Predicate<ProfileRequestContext> predicate) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        consentEnabledPredicate =
                Constraint.isNotNull(predicate, "predicate used to check if consent is enabled cannot be null");
    }
    
    /**
     * Set the lookup function to supply strategy bi-function for manipulating token claims set.
     * 
     * @param strategy What to set
     */
    public void setTokenClaimsSetManipulationStrategyLookupStrategy(@Nonnull final
            Function<ProfileRequestContext,BiFunction<ProfileRequestContext,Map<String,Object>,Map<String,Object>>>
            strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        tokenClaimsSetManipulationStrategyLookupStrategy =
                Constraint.isNotNull(strategy, "Manipulation strategy lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (dataSealer == null) {
            throw new ComponentInitializationException("DataSealer cannot be null");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        subjectCtx = profileRequestContext.getSubcontext(SubjectContext.class, false);
        if (subjectCtx == null) {
            log.warn("{} No subject context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        
        idGenerator = idGeneratorLookupStrategy.apply(profileRequestContext);
        if (idGenerator == null) {
            log.warn("{} No identifier generation strategy", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_SEC_CFG);
            return false;
        }
        
        authzCodeLifetime = authzCodeLifetimeLookupStrategy.apply(profileRequestContext);
        if (authzCodeLifetime == null) {
            log.warn("{} No authorization code lifetime available", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
            return false;
        }
        
        codeChallenge = codeChallengeLookupStrategy.apply(profileRequestContext);
        if (codeChallenge != null && !codeChallenge.isEmpty()) {
            // Prepend the PKCE challenge with method as we need to store both method and challenge.
            // Options for method are "plain" and "S256"
            final String codeChallengeMethod = codeChallengeMethodLookupStrategy.apply(profileRequestContext);
            // Default method is "plain"
            codeChallenge = (codeChallengeMethod != null ? codeChallengeMethod : "plain") + codeChallenge;
        }

        manipulationStrategy = tokenClaimsSetManipulationStrategyLookupStrategy.apply(profileRequestContext);

        return true;
    }

 // Checkstyle: MethodLength OFF

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final OIDCAuthenticationResponseContext responseCtx = getOidcResponseContext();
        
        final OIDCAuthenticationResponseConsentContext consentCtx =
                consentContextLookupStrategy.apply(profileRequestContext);
        final JSONArray consented = consentCtx != null ? consentCtx.getConsentedAttributes() : null;
        ClaimsSet claims = null;
        ClaimsSet claimsID = null;
        ClaimsSet claimsUI = null;
        final OIDCAuthenticationResponseTokenClaimsContext tokenClaimsCtx =
                tokenClaimsContextLookupStrategy.apply(profileRequestContext);
        if (tokenClaimsCtx != null) {
            claims = tokenClaimsCtx.getClaims();
            claimsID = tokenClaimsCtx.getIdtokenClaims();
            claimsUI = tokenClaimsCtx.getUserinfoClaims();
        }
        final Instant dateExp = Instant.now().plus(authzCodeLifetime);
        final Scope scope = responseCtx.getScope();
        final AuthorizeCodeClaimsSet claimsSet = new AuthorizeCodeClaimsSet.Builder()
                .setJWTID(idGenerator)
                .setClientID(getAuthenticationRequest().getClientID())
                .setIssuer(issuerLookupStrategy.apply(profileRequestContext))
                .setPrincipal(subjectCtx.getPrincipalName())
                .setSubject(responseCtx.getSubject())
                .setIssuedAt(Instant.now())
                .setExpiresAt(dateExp)
                .setAuthenticationTime(responseCtx.getAuthTime())
                .setRedirectURI(responseCtx.getRedirectURI())
                .setScope(scope != null ? scope : new Scope())
                .setAudience(responseCtx.getAudience())
                .setACR(responseCtx.getAcr())
                .setNonce(new DefaultRequestNonceLookupFunction().apply(profileRequestContext))
                .setCodeChallenge(codeChallenge)
                .setClaimsRequest(responseCtx.getRequestedClaims())
                .setDlClaims(claims)
                .setDlClaimsID(claimsID)
                .setDlClaimsUI(claimsUI)
                .setConsentedClaims(consented)
                .setConsentEnabled(consentEnabledPredicate.test(profileRequestContext))
                .build();

        if (manipulationStrategy != null) {
            log.debug("{} Manipulation strategy has been set, applying it to the claims set {}", getLogPrefix(),
                    claimsSet.serialize());
            final Map<String, Object> result = manipulationStrategy.apply(profileRequestContext,
                    claimsSet.getClaimsSet().toJSONObject());
            if (result == null) {
                log.debug("{} Manipulation strategy returned null, leaving token claims set untouched.",
                        getLogPrefix());
            } else {
                log.debug("{} Applying the manipulated claims into the token claims set", getLogPrefix());
                try {
                    claimsSet.setClaimsSet(JWTClaimsSet.parse(result));
                } catch (final ParseException e) {
                    log.error("{} The resulted claims set could not be transformed into ", getLogPrefix(), e);
                    ActionSupport.buildEvent(profileRequestContext, IdPEventIds.INVALID_PROFILE_CONFIG);
                    return;
                }
            }
        } else {
            log.debug("{} No manipulation strategy configured", getLogPrefix());
        }

        // We set token claims set to response context for possible access token generation.
        responseCtx.setAuthorizationGrantClaimsSet(claimsSet);
        try {
            responseCtx.setAuthorizationCode(claimsSet.serialize(dataSealer));
            log.debug("{} Setting authz code {} as {} to response context ", getLogPrefix(), claimsSet.serialize(),
                    responseCtx.getAuthorizationCode());
        } catch (final DataSealerException e) {
            log.error("{} Authorization Code generation failed {}", getLogPrefix(), e.getMessage());
            ActionSupport.buildEvent(profileRequestContext, EventIds.UNABLE_TO_ENCRYPT);
        }
    }

 // Checkstyle: MethodLength ON

}