/*
 * Licensed to the University Corporation for Advanced Internet Development,
 * Inc. (UCAID) under one or more contributor license agreements.  See the
 * NOTICE file distributed with this work for additional information regarding
 * copyright ownership. The UCAID licenses this file to You under the Apache
 * License, Version 2.0 (the "License"); you may not use this file except in
 * compliance with the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.plugin.oidc.op.profile.impl;

import net.shibboleth.idp.profile.IdPEventIds;
import net.shibboleth.idp.profile.context.RelyingPartyContext;
import net.shibboleth.idp.saml.profile.context.navigate.SAMLMetadataContextLookupFunction;
import net.shibboleth.oidc.metadata.context.OIDCMetadataContext;
import net.shibboleth.utilities.java.support.component.ComponentSupport;
import net.shibboleth.utilities.java.support.logic.Constraint;

import java.net.URI;
import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.MessageContext;
import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.opensaml.saml.common.messaging.context.SAMLMetadataContext;
import org.opensaml.saml.ext.saml2mdui.DisplayName;
import org.opensaml.saml.ext.saml2mdui.InformationURL;
import org.opensaml.saml.ext.saml2mdui.Logo;
import org.opensaml.saml.ext.saml2mdui.PrivacyStatementURL;
import org.opensaml.saml.ext.saml2mdui.UIInfo;
import org.opensaml.saml.ext.saml2mdui.impl.DisplayNameBuilder;
import org.opensaml.saml.ext.saml2mdui.impl.InformationURLBuilder;
import org.opensaml.saml.ext.saml2mdui.impl.LogoBuilder;
import org.opensaml.saml.ext.saml2mdui.impl.PrivacyStatementURLBuilder;
import org.opensaml.saml.ext.saml2mdui.impl.UIInfoBuilder;
import org.opensaml.saml.saml2.metadata.ContactPerson;
import org.opensaml.saml.saml2.metadata.ContactPersonTypeEnumeration;
import org.opensaml.saml.saml2.metadata.EmailAddress;
import org.opensaml.saml.saml2.metadata.EntityDescriptor;
import org.opensaml.saml.saml2.metadata.Extensions;
import org.opensaml.saml.saml2.metadata.SPSSODescriptor;
import org.opensaml.saml.saml2.metadata.impl.ContactPersonBuilder;
import org.opensaml.saml.saml2.metadata.impl.EmailAddressBuilder;
import org.opensaml.saml.saml2.metadata.impl.EntityDescriptorBuilder;
import org.opensaml.saml.saml2.metadata.impl.ExtensionsBuilder;
import org.opensaml.saml.saml2.metadata.impl.SPSSODescriptorBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nimbusds.langtag.LangTag;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

/**
 * Action that adds an outbound {@link MessageContext} and related OIDC contexts to the {@link ProfileRequestContext}
 * based on the identity of a relying party accessed via a lookup strategy, by default an immediate child of the profile
 * request context.
 * 
 * This action also initializes the {@link SAMLMetadataContext} and populates it with service and {@link UIInfo}
 * -related data.
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 * @event {@link IdPEventIds#INVALID_RELYING_PARTY_CTX}
 * @event {@link EventIds#INVALID_MSG_CTX}
 */
public class InitializeOutboundAuthenticationResponseMessageContext
        extends AbstractInitializeOutboundResponseMessageContext {

    /** Class logger. */
    @Nonnull
    private final Logger log = LoggerFactory.getLogger(InitializeOutboundAuthenticationResponseMessageContext.class);

    /** Strategy function to lookup the {@link OIDCMetadataContext}. */
    @Nonnull private Function<ProfileRequestContext, OIDCMetadataContext> oidcMetadataCtxLookupStrategy;
    
    /**
     * Strategy used to locate the {@link RelyingPartyContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext, RelyingPartyContext> relyingPartyCtxLookupStrategy;

    /**
     * Strategy used to locate the {@link SAMLMetadataContext} associated with a given {@link ProfileRequestContext}.
     */
    @Nonnull private Function<ProfileRequestContext, SAMLMetadataContext> samlMetadataCtxLookupStrategy;

    /** The OIDC metadata context used as a source for the SAML metadata context. */
    private OIDCMetadataContext oidcMetadataCtx;
    
    /** The relying party context used for storing the SAML metadata context. */
    private RelyingPartyContext relyingPartyCtx;

    /** The default language when it has not been defined in the metadata. */
    private String defaultLanguage;

    /**
     * Constructor.
     */
    public InitializeOutboundAuthenticationResponseMessageContext() {
        oidcMetadataCtxLookupStrategy = new ChildContextLookup<>(OIDCMetadataContext.class).compose(
                new InboundMessageContextLookup());
        relyingPartyCtxLookupStrategy = new ChildContextLookup<>(RelyingPartyContext.class);
        samlMetadataCtxLookupStrategy = new SAMLMetadataContextLookupFunction();
        defaultLanguage = "en";
    }

    /**
     * Get the mechanism to lookup the {@link OIDCMetadataContext} from the {@link ProfileRequestContext}.
     * 
     * @return The mechanism to lookup the {@link OIDCMetadataContext} from the {@link ProfileRequestContext}.
     */
    @Nonnull
    public Function<ProfileRequestContext, OIDCMetadataContext> getOIDCMetadataContextLookupStrategy() {
        return oidcMetadataCtxLookupStrategy;
    }

    /**
     * Set the mechanism to lookup the {@link OIDCMetadataContext} from the {@link ProfileRequestContext}.
     * 
     * @param strgy What to set.
     */
    public void setOIDCMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, OIDCMetadataContext> strgy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        oidcMetadataCtxLookupStrategy = Constraint.isNotNull(strgy, "Injected Metadata Strategy cannot be null");
    }

    /**
     * Get the mechanism to lookup the {@link RelyingPartyContext} from the {@link ProfileRequestContext}.
     * 
     * @return The mechanism to lookup the {@link RelyingPartyContext} from the {@link ProfileRequestContext}.
     */
    @Nonnull
    public Function<ProfileRequestContext, RelyingPartyContext> getRelyingPartyContextLookupStrategy() {
        return relyingPartyCtxLookupStrategy;
    }

    /**
     * Set the strategy used to locate the {@link RelyingPartyContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link RelyingPartyContext} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setRelyingPartyContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, RelyingPartyContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        relyingPartyCtxLookupStrategy =
                Constraint.isNotNull(strategy, "RelyingPartyContext lookup strategy cannot be null");
    }

    /**
     * Get the mechanism to lookup the {@link SAMLMetadataContext} from the {@link ProfileRequestContext}.
     * 
     * @return The mechanism to lookup the {@link SAMLMetadataContext} from the {@link ProfileRequestContext}.
     */
    @Nonnull
    public Function<ProfileRequestContext, SAMLMetadataContext> getSAMLMetadataContextLookupStrategy() {
        return samlMetadataCtxLookupStrategy;
    }

    /**
     * Set the strategy used to locate the {@link SAMLMetadataContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy strategy used to locate the {@link SAMLMetadataContext} associated with a given
     *            {@link ProfileRequestContext}
     */
    public void setSAMLMetadataContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext, SAMLMetadataContext> strategy) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        samlMetadataCtxLookupStrategy =
                Constraint.isNotNull(strategy, "SAMLMetadataContext lookup strategy cannot be null");
    }

    /**
     * Set the default language when it has not been defined in the metadata.
     * 
     * @param language What to set.
     */
    public void setDefaultLanguage(@Nonnull final String language) {
        ComponentSupport.ifInitializedThrowUnmodifiabledComponentException(this);

        defaultLanguage = Constraint.isNotEmpty(language, "The default language cannot be empty");
    }

    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        oidcMetadataCtx = oidcMetadataCtxLookupStrategy.apply(profileRequestContext);
        if (oidcMetadataCtx == null) {
            log.error("{} No OIDC metadata context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_MSG_CTX);
            return false;
        }
        
        relyingPartyCtx = relyingPartyCtxLookupStrategy.apply(profileRequestContext);
        if (relyingPartyCtx == null) {
            log.error("{} No relying party context", getLogPrefix());
            ActionSupport.buildEvent(profileRequestContext, EventIds.INVALID_PROFILE_CTX);
            return false;
        }
        
        return true;
    }

    // Checkstyle: CyclomaticComplexity OFF

    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        super.doExecute(profileRequestContext);

        if (samlMetadataCtxLookupStrategy.apply(profileRequestContext) != null) {
            log.debug("{} SAML metadata context already found", getLogPrefix());
            return;
        }

        final SAMLMetadataContext samlContext = new SAMLMetadataContext();
        final EntityDescriptor entityDescriptor = new EntityDescriptorBuilder().buildObject();
        entityDescriptor.setEntityID(oidcMetadataCtx.getClientInformation().getID().getValue());
        final OIDCClientMetadata oidcMetadata = oidcMetadataCtx.getClientInformation().getOIDCMetadata();
        final SPSSODescriptor spDescriptor = new SPSSODescriptorBuilder().buildObject();
        final UIInfo uiInfo = new UIInfoBuilder().buildObject();
        for (final LangTag tag : oidcMetadata.getLogoURIEntries().keySet()) {
            final Logo logo = new LogoBuilder().buildObject();
            logo.setXMLLang(tag == null ? defaultLanguage : tag.getLanguage());
            final URI logoUri = oidcMetadata.getLogoURI(tag);
            if (logoUri != null) {
                logo.setURI(logoUri.toString());
                uiInfo.getLogos().add(logo);
            }
        }
        for (final LangTag tag : oidcMetadata.getPolicyURIEntries().keySet()) {
            final PrivacyStatementURL url = new PrivacyStatementURLBuilder().buildObject();
            url.setXMLLang(tag == null ? defaultLanguage : tag.getLanguage());
            url.setURI(oidcMetadata.getPolicyURI(tag).toString());
            uiInfo.getPrivacyStatementURLs().add(url);
        }
        for (final LangTag tag : oidcMetadata.getTermsOfServiceURIEntries().keySet()) {
            final InformationURL url = new InformationURLBuilder().buildObject();
            url.setXMLLang(tag == null ? defaultLanguage : tag.getLanguage());
            url.setURI(oidcMetadata.getTermsOfServiceURI(tag).toString());
            uiInfo.getInformationURLs().add(url);
        }
        final List<String> emails = oidcMetadata.getEmailContacts();
        if (emails != null) {
            for (final String email : emails) {
                final ContactPerson contactPerson = new ContactPersonBuilder().buildObject();
                // TODO: should it be configurable?
                contactPerson.setType(ContactPersonTypeEnumeration.SUPPORT);
                final EmailAddress address = new EmailAddressBuilder().buildObject();
                address.setURI(email.startsWith("mailto:") ? email : "mailto:" + email);
                contactPerson.getEmailAddresses().add(address);
                entityDescriptor.getContactPersons().add(contactPerson);
            }
        }
        for (final LangTag tag : oidcMetadata.getNameEntries().keySet()) {
            final DisplayName displayName = new DisplayNameBuilder().buildObject();
            displayName.setXMLLang(tag == null ? defaultLanguage : tag.getLanguage());
            displayName.setValue(oidcMetadata.getNameEntries().get(tag));
            uiInfo.getDisplayNames().add(displayName);
        }
        final Extensions extensions = new ExtensionsBuilder().buildObject();
        extensions.getUnknownXMLObjects().add(uiInfo);
        spDescriptor.setExtensions(extensions);
        samlContext.setEntityDescriptor(entityDescriptor);
        samlContext.setRoleDescriptor(spDescriptor);

        relyingPartyCtx.setRelyingPartyIdContextTree(samlContext);
    }
    
    // Checkstyle: CyclomaticComplexity ON
    
}